<?php
/**
 * Plugin Name: Royal Moodle Sync
 * Plugin URI: https://veeraweb.com
 * Description: همگام‌سازی ساده بین Moodle و WordPress بدون نیاز به لایسنس - همگام‌سازی کاربران، دوره‌ها و ثبت‌نام‌ها
 * Version: 1.0.0
 * Author: saeed modarresi
 * Author URI: https://veeraweb.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: royal-moodle-sync
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// تعریف ثابت‌ها
define('MWS_VERSION', '1.0.0');
define('MWS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MWS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MWS_PLUGIN_BASENAME', plugin_basename(__FILE__));

// بارگذاری کلاس‌ها به صورت امن
function mws_load_classes() {
    static $loaded = false;
    
    if ($loaded) {
        return;
    }
    
    $includes_dir = MWS_PLUGIN_DIR . 'includes/';
    
    $classes = array(
        'class-moodle-api.php',
        'class-user-sync.php',
        'class-course-sync.php',
        'class-category-sync.php',
        'class-enrollment.php',
        'class-admin.php',
        'class-orders.php',
        'class-user-enrollment.php',
        'class-course-students.php',
        'class-sso.php'
    );
    
    foreach ($classes as $class_file) {
        $file_path = $includes_dir . $class_file;
        if (file_exists($file_path)) {
            require_once $file_path;
        }
    }
    
    // بارگذاری ادغام WooCommerce (اگر نصب باشد)
    if (class_exists('WooCommerce')) {
        $woo_file = $includes_dir . 'class-woocommerce-integration.php';
        if (file_exists($woo_file)) {
            require_once $woo_file;
        }
    }
    
    $loaded = true;
}

// بارگذاری کلاس‌ها فقط در صورت نیاز (نه در زمان بارگذاری فایل)
// mws_load_classes(); // این خط را کامنت می‌کنیم تا در activate لود شود

/**
 * کلاس اصلی پلاگین
 */
class Moodle_WP_Sync {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init_hooks();
    }
    
    private function init_hooks() {
        // فعال‌سازی پلاگین
        register_activation_hook(__FILE__, array($this, 'activate'));
        
        // غیرفعال‌سازی پلاگین
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // بارگذاری پلاگین
        add_action('plugins_loaded', array($this, 'init'));
        
        // بارگذاری ترجمه
        add_action('plugins_loaded', array($this, 'load_textdomain'));
    }
    
    /**
     * فعال‌سازی پلاگین
     */
    public function activate() {
        // بارگذاری کلاس‌ها
        mws_load_classes();
        
        // ایجاد جداول مورد نیاز
        $this->create_tables();
        
        // تنظیمات پیش‌فرض
        if (!get_option('mws_moodle_url')) {
            add_option('mws_moodle_url', '');
        }
        if (!get_option('mws_moodle_token')) {
            add_option('mws_moodle_token', '');
        }
        if (!get_option('mws_auto_sync_users')) {
            add_option('mws_auto_sync_users', '1');
        }
        if (!get_option('mws_auto_sync_courses')) {
            add_option('mws_auto_sync_courses', '1');
        }
        if (!get_option('mws_sync_batch_size')) {
            add_option('mws_sync_batch_size', '100');
        }
        
        // ثبت Custom Post Type
        $this->register_post_types();
        
        // ثبت Taxonomy دسته‌بندی‌ها
        $this->register_taxonomy_activate();
        
        flush_rewrite_rules();
        
        // سینک اولیه را به بعد از بارگذاری کامل پلاگین موکول می‌کنیم
        // این کار در init انجام می‌شود
    }
    
    /**
     * ثبت Taxonomy برای دسته‌بندی‌ها (برای استفاده در activate)
     */
    private function register_taxonomy_activate() {
        $labels = array(
            'name' => 'دسته‌بندی‌های Moodle',
            'singular_name' => 'دسته‌بندی Moodle',
            'menu_name' => 'دسته‌بندی‌های Moodle',
            'all_items' => 'همه دسته‌بندی‌ها',
            'edit_item' => 'ویرایش دسته‌بندی',
            'view_item' => 'مشاهده دسته‌بندی',
            'update_item' => 'به‌روزرسانی دسته‌بندی',
            'add_new_item' => 'افزودن دسته‌بندی جدید',
            'new_item_name' => 'نام دسته‌بندی جدید',
            'search_items' => 'جستجوی دسته‌بندی‌ها'
        );
        
        register_taxonomy('moodle_category', array('moodle_course'), array(
            'labels' => $labels,
            'hierarchical' => true,
            'public' => true,
            'show_ui' => true,
            'show_admin_column' => true,
            'show_in_rest' => true,
            'rewrite' => array('slug' => 'moodle-category')
        ));
    }
    
    /**
     * غیرفعال‌سازی پلاگین
     */
    public function deactivate() {
        // حذف کرون جاب‌ها
        wp_clear_scheduled_hook('mws_sync_courses_cron');
        flush_rewrite_rules();
    }
    
    /**
     * ایجاد جداول دیتابیس
     */
    private function create_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        
        // جدول لاگ همگام‌سازی
        $table_name = $wpdb->prefix . 'mws_sync_log';
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            user_id bigint(20) NOT NULL,
            moodle_user_id bigint(20) DEFAULT NULL,
            course_id bigint(20) DEFAULT NULL,
            action varchar(50) NOT NULL,
            status varchar(20) NOT NULL,
            message text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            KEY user_id (user_id),
            KEY moodle_user_id (moodle_user_id),
            KEY course_id (course_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * راه‌اندازی پلاگین
     */
    public function init() {
        // اطمینان از بارگذاری کلاس‌ها
        if (!class_exists('MWS_Moodle_API')) {
            mws_load_classes();
        }
        
        // ثبت Custom Post Type
        add_action('init', array($this, 'register_post_types'));
        
        // ثبت Taxonomy (اگر قبلاً ثبت نشده باشد)
        add_action('init', array($this, 'register_taxonomy'));
        
        // بارگذاری کلاس‌های اصلی با بررسی وجود کلاس
        try {
            if (is_admin() && class_exists('MWS_Admin')) {
                new MWS_Admin();
            }
            
            // همگام‌سازی کاربران
            if (class_exists('MWS_User_Sync')) {
                $user_sync = new MWS_User_Sync();
                $user_sync->init();
            }
            
            // همگام‌سازی دوره‌ها
            if (class_exists('MWS_Course_Sync')) {
                $course_sync = new MWS_Course_Sync();
                $course_sync->init();
            }
            
            // همگام‌سازی دسته‌بندی‌ها
            if (class_exists('MWS_Category_Sync')) {
                $category_sync = new MWS_Category_Sync();
                $category_sync->init();
            }
            
            // مدیریت ثبت‌نام
            if (class_exists('MWS_Enrollment')) {
                $enrollment = new MWS_Enrollment();
                $enrollment->init();
            }
            
            // مدیریت سفارشات
            if (class_exists('WooCommerce') && class_exists('MWS_Orders')) {
                new MWS_Orders();
            }
            
            // مدیریت اتصال کاربران (AJAX handlers)
            if (class_exists('MWS_User_Enrollment_Manager')) {
                $enrollment_manager = new MWS_User_Enrollment_Manager();
                $enrollment_manager->init();
            }

            // متاباکس دانشجویان + پروفایل کاربر
            if (class_exists('MWS_Course_Students')) {
                $course_students = new MWS_Course_Students();
                $course_students->init();
            }
            
            // SSO
            if (class_exists('MWS_SSO')) {
                $sso = new MWS_SSO();
                $sso->init();
            }
        } catch (Exception $e) {
            error_log('MWS Init Error: ' . $e->getMessage());
        }
        
        // بارگذاری استایل‌ها و اسکریپت‌ها
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        
        // سینک اولیه (اگر Moodle تنظیم شده باشد و اولین بار است)
        if (get_option('mws_first_activation') !== 'done') {
            $this->maybe_initial_sync();
            update_option('mws_first_activation', 'done');
        }
    }
    
    /**
     * ثبت Taxonomy (برای استفاده در init)
     */
    public function register_taxonomy() {
        if (taxonomy_exists('moodle_category')) {
            return;
        }
        
        $labels = array(
            'name' => 'دسته‌بندی‌های Moodle',
            'singular_name' => 'دسته‌بندی Moodle',
            'menu_name' => 'دسته‌بندی‌های Moodle',
            'all_items' => 'همه دسته‌بندی‌ها',
            'edit_item' => 'ویرایش دسته‌بندی',
            'view_item' => 'مشاهده دسته‌بندی',
            'update_item' => 'به‌روزرسانی دسته‌بندی',
            'add_new_item' => 'افزودن دسته‌بندی جدید',
            'new_item_name' => 'نام دسته‌بندی جدید',
            'search_items' => 'جستجوی دسته‌بندی‌ها'
        );
        
        register_taxonomy('moodle_category', array('moodle_course'), array(
            'labels' => $labels,
            'hierarchical' => true,
            'public' => true,
            'show_ui' => true,
            'show_admin_column' => true,
            'show_in_rest' => true,
            'rewrite' => array('slug' => 'moodle-category')
        ));
    }
    
    /**
     * سینک اولیه (اگر ممکن باشد)
     */
    private function maybe_initial_sync() {
        $moodle_url = get_option('mws_moodle_url');
        $moodle_token = get_option('mws_moodle_token');
        
        if (empty($moodle_url) || empty($moodle_token)) {
            return;
        }
        
        // سینک اولیه را به صورت غیرهمزمان انجام می‌دهیم
        // تا از خطا جلوگیری کنیم
        try {
            // سینک دسته‌بندی‌ها
            $category_sync = new MWS_Category_Sync();
            $category_sync->sync_categories();
            
            // سینک دوره‌ها
            $course_sync = new MWS_Course_Sync();
            $course_sync->sync_courses();
        } catch (Exception $e) {
            // خطا را لاگ می‌کنیم اما فعال‌سازی را متوقف نمی‌کنیم
            error_log('MWS Initial Sync Error: ' . $e->getMessage());
        }
    }
    
    /**
     * ثبت Custom Post Type (برای استفاده در init)
     */
    public function register_post_types() {
        register_post_type('moodle_course', array(
            'labels' => array(
                'name' => 'دوره‌های Moodle',
                'singular_name' => 'دوره Moodle',
                'add_new' => 'افزودن دوره',
                'add_new_item' => 'افزودن دوره جدید',
                'edit_item' => 'ویرایش دوره',
                'new_item' => 'دوره جدید',
                'view_item' => 'مشاهده دوره',
                'search_items' => 'جستجوی دوره‌ها',
                'not_found' => 'دوره‌ای یافت نشد',
                'not_found_in_trash' => 'دوره‌ای در سطل زباله یافت نشد'
            ),
            'public' => true,
            'has_archive' => true,
            'rewrite' => array('slug' => 'moodle-courses'),
            'supports' => array('title', 'editor', 'thumbnail', 'excerpt'),
            'menu_icon' => 'dashicons-book-alt',
            'show_in_rest' => true,
            'show_in_menu' => false, // مخفی کردن منوی جداگانه
            'show_ui' => true
        ));
    }
    
    /**
     * بارگذاری ترجمه
     */
    public function load_textdomain() {
        load_plugin_textdomain('royal-moodle-sync', false, dirname(MWS_PLUGIN_BASENAME) . '/languages');
    }
    
    /**
     * بارگذاری فایل‌های ادمین
     */
    public function enqueue_admin_assets($hook) {
        $is_moodle_course_page = (
            ($hook === 'post.php'     && isset($_GET['post'])      && get_post_type($_GET['post']) === 'moodle_course') ||
            ($hook === 'post-new.php' && isset($_GET['post_type']) && $_GET['post_type']           === 'moodle_course')
        );

        $is_plugin_page = (
            strpos($hook, 'royal-moodle-sync') !== false ||
            strpos($hook, 'moodle-wp-sync') !== false ||
            ($hook === 'edit.php' && isset($_GET['post_type']) && $_GET['post_type'] === 'moodle_course') ||
            $is_moodle_course_page ||
            ($hook === 'edit-tags.php' && isset($_GET['taxonomy']) && $_GET['taxonomy'] === 'moodle_category') ||
            $hook === 'profile.php' ||
            $hook === 'user-edit.php'
        );

        if (!$is_plugin_page) {
            return;
        }

        wp_enqueue_style('mws-admin-style', MWS_PLUGIN_URL . 'assets/admin.css', array(), MWS_VERSION);

        // Select2 برای صفحه ویرایش دوره (جستجوی کاربر در متاباکس)
        if ($is_moodle_course_page) {
            wp_enqueue_script('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js', array('jquery'), '4.1.0', true);
            wp_enqueue_style('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css', array(), '4.1.0');
        }

        wp_enqueue_script('mws-admin-script', MWS_PLUGIN_URL . 'assets/admin.js', array('jquery'), MWS_VERSION, true);

        wp_localize_script('mws-admin-script', 'mwsAjax', array(
            'ajaxurl'    => admin_url('admin-ajax.php'),
            'nonce'      => wp_create_nonce('mws_ajax_nonce'),
            'batch_size' => (int) get_option('mws_sync_batch_size', 100),
        ));
    }
    
    /**
     * بارگذاری فایل‌های فرانت‌اند
     */
    public function enqueue_frontend_assets() {
        wp_enqueue_style('mws-frontend-style', MWS_PLUGIN_URL . 'assets/frontend.css', array(), MWS_VERSION);
    }
}

// راه‌اندازی پلاگین
Moodle_WP_Sync::get_instance();

