<?php
/**
 * کلاس ادغام با WooCommerce
 * این کلاس به صورت اختیاری استفاده می‌شود
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_WooCommerce_Integration {
    
    public function __construct() {
        // فقط اگر WooCommerce نصب باشد
        if (class_exists('WooCommerce')) {
            $this->init();
        }
    }
    
    /**
     * راه‌اندازی هوک‌ها
     */
    public function init() {
        // افزودن تب جدید در صفحه ویرایش محصول
        add_filter('woocommerce_product_data_tabs', array($this, 'add_course_tab'), 99);
        
        // محتوای تب
        add_action('woocommerce_product_data_panels', array($this, 'add_course_tab_content'));
        
        // ذخیره فیلد متا
        add_action('woocommerce_process_product_meta', array($this, 'save_course_field'));
        
        // بارگذاری Select2 و اسکریپت‌ها
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // AJAX handler برای جستجوی دوره‌ها
        add_action('wp_ajax_mws_search_courses_for_product', array($this, 'ajax_search_courses'));

        // به‌روزرسانی خودکار قیمت محصول ووکامرس وقتی قیمت دوره عوض شد
        add_action('save_post_moodle_course', array($this, 'sync_course_price_to_products'), 25);
    }
    
    /**
     * افزودن تب جدید
     */
    public function add_course_tab($tabs) {
        $tabs['mws_course_sync'] = array(
            'label'    => 'سینک مودل رویال',
            'target'   => 'mws_course_sync_data',
            'class'    => array('show_if_simple', 'show_if_variable'),
            'priority' => 25,
        );
        return $tabs;
    }
    
    /**
     * محتوای تب
     */
    public function add_course_tab_content() {
        global $post;
        
        $current_course_id = get_post_meta($post->ID, 'mws_linked_course_id', true);
        $current_course_post_id = 0;
        
        // پیدا کردن post ID دوره مرتبط
        if ($current_course_id) {
            $courses = get_posts(array(
                'post_type' => 'moodle_course',
                'meta_query' => array(
                    array(
                        'key' => 'mws_moodle_course_id',
                        'value' => $current_course_id,
                        'compare' => '='
                    )
                ),
                'posts_per_page' => 1,
                'fields' => 'ids'
            ));
            
            if (!empty($courses)) {
                $current_course_post_id = $courses[0];
            }
        }
        
        ?>
        <div id="mws_course_sync_data" class="panel woocommerce_options_panel">
            <div class="options_group">
                <p class="form-field">
                    <label for="mws_course_selector">انتخاب دوره:</label>
                    <select id="mws_course_selector" name="mws_linked_course_id" class="mws-select2-course" style="width: 100%;">
                        <?php if ($current_course_post_id) : 
                            $course = get_post($current_course_post_id);
                            $moodle_id = get_post_meta($current_course_post_id, 'mws_moodle_course_id', true);
                            $course_price = get_post_meta($current_course_post_id, 'mws_course_price', true);
                            $course_type = get_post_meta($current_course_post_id, 'mws_course_type', true);
                        ?>
                            <option value="<?php echo esc_attr($moodle_id); ?>" 
                                    data-post-id="<?php echo esc_attr($current_course_post_id); ?>"
                                    data-price="<?php echo esc_attr($course_price); ?>"
                                    data-type="<?php echo esc_attr($course_type); ?>"
                                    selected>
                                <?php echo esc_html($course->post_title . ' (ID: ' . $moodle_id . ')'); ?>
                            </option>
                        <?php else : ?>
                            <option value="">-- جستجو و انتخاب دوره --</option>
                        <?php endif; ?>
                    </select>
                    <span class="description">
                        برای جستجو، شروع به تایپ کنید. با انتخاب دوره، قیمت محصول به صورت خودکار تنظیم می‌شود.
                    </span>
                </p>
                
                <p class="form-field" id="mws_course_info" style="display: none;">
                    <label>اطلاعات دوره انتخاب شده:</label>
                    <span id="mws_course_info_content"></span>
                </p>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // اطمینان از وجود ajaxurl
            if (typeof ajaxurl === 'undefined') {
                var ajaxurl = '<?php echo admin_url('admin-ajax.php'); ?>';
            }
            
            // Initialize Select2 for course selector
            if ($.fn.select2) {
                $('#mws_course_selector').select2({
                    placeholder: 'جستجو و انتخاب دوره...',
                    allowClear: true,
                    minimumInputLength: 0,
                    ajax: {
                        url: ajaxurl,
                        dataType: 'json',
                        delay: 250,
                        data: function(params) {
                            return {
                                q: params.term || '',
                                page: params.page || 1,
                                action: 'mws_search_courses_for_product',
                                nonce: '<?php echo wp_create_nonce('mws_ajax_nonce'); ?>'
                            };
                        },
                        processResults: function(data) {
                            return {
                                results: data.results || [],
                                pagination: {
                                    more: data.pagination && data.pagination.more
                                }
                            };
                        },
                        cache: true
                    },
                    language: {
                        noResults: function() {
                            return 'نتیجه‌ای یافت نشد';
                        },
                        searching: function() {
                            return 'در حال جستجو...';
                        },
                        inputTooShort: function() {
                            return 'حداقل 0 کاراکتر وارد کنید';
                        }
                    }
                });
                
                // وقتی دوره انتخاب شد
                $('#mws_course_selector').on('select2:select', function(e) {
                    var data = e.params.data;
                    var moodleId = data.id;
                    var postId = data.post_id || '';
                    var price = data.price || '0';
                    var courseType = data.type || '';
                    
                    // نمایش اطلاعات دوره
                    if (moodleId) {
                        var infoHtml = '<strong>شناسه Moodle:</strong> ' + moodleId;
                        if (price && parseFloat(price) > 0) {
                            infoHtml += ' | <strong>قیمت:</strong> ' + parseFloat(price).toLocaleString('fa-IR') + ' تومان';
                        } else {
                            infoHtml += ' | <strong>نوع:</strong> رایگان';
                        }
                        $('#mws_course_info_content').html(infoHtml);
                        $('#mws_course_info').show();
                        
                        // تنظیم قیمت محصول
                        var productPrice = parseFloat(price) || 0;
                        $('#_regular_price').val(productPrice).trigger('change');
                        $('#_sale_price').val('');
                        
                        // اگر دوره رایگان است
                        if (productPrice == 0 || courseType == 'free') {
                            $('#_regular_price').val('0');
                            $('#_sale_price').val('');
                        }
                    } else {
                        $('#mws_course_info').hide();
                    }
                });
                
                // وقتی دوره حذف شد
                $('#mws_course_selector').on('select2:clear', function() {
                    $('#mws_course_info').hide();
                    $('#mws_linked_course_id').val('');
                });
            }
        });
        </script>
        <?php
    }
    
    /**
     * AJAX handler برای جستجوی دوره‌ها
     */
    public function ajax_search_courses() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $search = isset($_GET['q']) ? sanitize_text_field($_GET['q']) : '';
        $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
        $per_page = 20;
        
        $args = array(
            'post_type' => 'moodle_course',
            'posts_per_page' => $per_page,
            'offset' => ($page - 1) * $per_page,
            'post_status' => 'publish',
            'orderby' => 'title',
            'order' => 'ASC',
            'fields' => 'ids',
        );
        
        if (!empty($search)) {
            $args['s'] = $search;
        }
        
        $course_query = new WP_Query($args);
        $courses = $course_query->posts;
        $total = $course_query->found_posts;
        
        $results = array();
        foreach ($courses as $course_id) {
            $course = get_post($course_id);
            $moodle_id = get_post_meta($course_id, 'mws_moodle_course_id', true);
            $course_price = get_post_meta($course_id, 'mws_course_price', true);
            $course_type = get_post_meta($course_id, 'mws_course_type', true);
            
            if ($moodle_id) {
                $price_text = '';
                if ($course_price && floatval($course_price) > 0) {
                    $price_text = ' - ' . number_format(floatval($course_price)) . ' تومان';
                } else {
                    $price_text = ' - رایگان';
                }
                
                $results[] = array(
                    'id' => $moodle_id,
                    'text' => $course->post_title . ' (ID: ' . $moodle_id . ')' . $price_text,
                    'post_id' => $course_id,
                    'price' => $course_price ? floatval($course_price) : 0,
                    'type' => $course_type ? $course_type : 'free'
                );
            }
        }
        
        wp_send_json(array(
            'results' => $results,
            'pagination' => array(
                'more' => ($page * $per_page) < $total
            )
        ));
    }
    
    /**
     * ذخیره فیلد متا و تنظیم قیمت
     */
    public function save_course_field($post_id) {
        $moodle_course_id = isset($_POST['mws_linked_course_id']) ? sanitize_text_field($_POST['mws_linked_course_id']) : '';
        
        if (!empty($moodle_course_id)) {
            update_post_meta($post_id, 'mws_linked_course_id', $moodle_course_id);
            
            // پیدا کردن دوره و تنظیم قیمت
            $courses = get_posts(array(
                'post_type' => 'moodle_course',
                'meta_query' => array(
                    array(
                        'key' => 'mws_moodle_course_id',
                        'value' => $moodle_course_id,
                        'compare' => '='
                    )
                ),
                'posts_per_page' => 1,
                'fields' => 'ids'
            ));
            
            if (!empty($courses)) {
                $course_post_id = $courses[0];
                $course_price = get_post_meta($course_post_id, 'mws_course_price', true);
                $course_type = get_post_meta($course_post_id, 'mws_course_type', true);
                
                // تنظیم قیمت محصول
                $product_price = 0;
                if ($course_price && floatval($course_price) > 0) {
                    $product_price = floatval($course_price);
                } else if ($course_type == 'free') {
                    $product_price = 0;
                }
                
                // ذخیره قیمت
                update_post_meta($post_id, '_regular_price', $product_price);
                update_post_meta($post_id, '_price', $product_price);
                
                // اگر رایگان است، sale price را پاک کن
                if ($product_price == 0) {
                    update_post_meta($post_id, '_sale_price', '');
                }
            }
        } else {
            delete_post_meta($post_id, 'mws_linked_course_id');
        }
    }
 
    /**
     * وقتی قیمت دوره تغییر کرد، قیمت محصول(های) ووکامرس وصل‌شده را به‌روز کن
     */
    public function sync_course_price_to_products($post_id) {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        if (wp_is_post_revision($post_id)) {
            return;
        }
        
        $moodle_id = get_post_meta($post_id, 'mws_moodle_course_id', true);
        if (empty($moodle_id)) {
            return;
        }
        
        $course_price = get_post_meta($post_id, 'mws_course_price', true);
        $course_type = get_post_meta($post_id, 'mws_course_type', true);
        
        $product_price = 0;
        if ($course_price && floatval($course_price) > 0) {
            $product_price = floatval($course_price);
        } elseif ($course_type === 'free') {
            $product_price = 0;
        }
        
        $products = get_posts(array(
            'post_type'      => 'product',
            'posts_per_page' => -1,
            'post_status'    => 'any',
            'fields'         => 'ids',
            'meta_query'     => array(
                array(
                    'key'   => 'mws_linked_course_id',
                    'value' => $moodle_id,
                    'compare' => '='
                )
            )
        ));
        
        foreach ($products as $product_id) {
            update_post_meta($product_id, '_regular_price', $product_price);
            update_post_meta($product_id, '_price', $product_price);
            if ($product_price == 0) {
                update_post_meta($product_id, '_sale_price', '');
            }
        }
    }
    
    /**
     * بارگذاری اسکریپت‌ها و استایل‌ها
     */
    public function enqueue_admin_scripts($hook) {
        // فقط در صفحه ویرایش محصول
        if ($hook != 'post.php' && $hook != 'post-new.php') {
            return;
        }
        
        global $post;
        if (!$post || $post->post_type != 'product') {
            return;
        }
        
        // بارگذاری Select2 (اگر قبلاً بارگذاری نشده باشد)
        if (!wp_script_is('select2', 'enqueued')) {
            wp_enqueue_style('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css', array(), '4.1.0-rc.0');
            wp_enqueue_script('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js', array('jquery'), '4.1.0-rc.0', true);
        }
        
        // استایل RTL برای Select2
        wp_enqueue_style('select2-rtl', MWS_PLUGIN_URL . 'assets/select2-rtl.css', array('select2'), MWS_VERSION);
        
        // اضافه کردن ajaxurl اگر وجود نداشته باشد
        wp_add_inline_script('select2', 'var ajaxurl = ajaxurl || "' . admin_url('admin-ajax.php') . '";', 'before');
    }
}

// راه‌اندازی ادغام WooCommerce
if (class_exists('WooCommerce')) {
    new MWS_WooCommerce_Integration();
}
