<?php
/**
 * کلاس همگام‌سازی کاربران بین WordPress و Moodle
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_User_Sync {
    private $moodle_api;
    
    public function __construct() {
        $this->moodle_api = new MWS_Moodle_API();
    }
    
    /**
     * راه‌اندازی هوک‌ها
     */
    public function init() {
        // همگام‌سازی هنگام ثبت‌نام کاربر جدید
        if (get_option('mws_auto_sync_users', '1') == '1') {
            add_action('user_register', array($this, 'sync_user_to_moodle'), 10, 1);
        }
        
        // همگام‌سازی هنگام به‌روزرسانی پروفایل
        add_action('profile_update', array($this, 'update_user_in_moodle'), 10, 2);
        
        // همگام‌سازی دستی از صفحه ادمین
        add_action('wp_ajax_mws_sync_user', array($this, 'ajax_sync_user'));
        
        // همگام‌سازی دسته‌ای
        add_action('wp_ajax_mws_sync_all_users', array($this, 'ajax_sync_all_users'));

        add_action('wp_ajax_mws_get_user_sync_stats', array($this, 'ajax_get_user_sync_stats'));
        
        // دریافت کاربران از Moodle
        add_action('wp_ajax_mws_import_users_from_moodle', array($this, 'ajax_import_users_from_moodle'));
    }
    
    /**
     * دریافت کاربران از Moodle و ایجاد در WordPress
     */
    public function import_users_from_moodle() {
        if (!$this->moodle_api->is_configured()) {
            return array('success' => false, 'message' => 'Moodle تنظیم نشده است');
        }
        
        // ابتدا بدون pagination همه کاربران را می‌گیریم
        // چون pagination در بعضی از نسخه‌های Moodle مشکل دارد
        $moodle_users = $this->moodle_api->get_all_users();
        
        // بررسی خطا
        if (is_wp_error($moodle_users)) {
            return array('success' => false, 'message' => 'خطا در دریافت کاربران از Moodle: ' . $moodle_users->get_error_message() . '. لطفاً مطمئن شوید که Web Service Token دسترسی به core_user_get_users دارد.');
        }
        
        if (!$moodle_users || !is_array($moodle_users) || count($moodle_users) == 0) {
            return array('success' => false, 'message' => 'هیچ کاربری از Moodle دریافت نشد. لطفاً مطمئن شوید که Web Service Token دسترسی به core_user_get_users دارد و در Moodle کاربری وجود دارد.');
        }
        
        $imported = 0;
        $updated = 0;
        $skipped = 0;
        $errors = array();
        
        foreach ($moodle_users as $moodle_user) {
            $moodle_user_id = isset($moodle_user['id']) ? intval($moodle_user['id']) : 0;
            $moodle_username = isset($moodle_user['username']) ? $moodle_user['username'] : 'Unknown';
            $moodle_email = isset($moodle_user['email']) ? $moodle_user['email'] : '';
            
            // رد کردن کاربران خاص Moodle
            if ($moodle_user_id > 0 && $moodle_user_id <= 2) {
                $skipped++;
                continue;
            }
            
            // بررسی وجود email
            if (empty($moodle_email)) {
                $skipped++;
                $errors[] = array(
                    'moodle_id' => $moodle_user_id,
                    'username' => $moodle_username,
                    'email' => '',
                    'reason' => 'بدون ایمیل'
                );
                continue;
            }
            
            // بررسی اینکه آیا کاربر با همین Moodle ID قبلاً import شده یا نه
            $existing_wp_users = get_users(array(
                'meta_key' => 'mws_moodle_user_id',
                'meta_value' => intval($moodle_user['id']),
                'number' => 1
            ));
            
            if (!empty($existing_wp_users)) {
                // کاربر قبلاً import شده است، فقط به‌روزرسانی می‌کنیم
                $wp_user = $existing_wp_users[0];
                update_user_meta($wp_user->ID, 'mws_synced_at', current_time('mysql'));
                $updated++;
                continue;
            }
            
            // بررسی اینکه آیا کاربر در WordPress وجود دارد (بر اساس email)
            $wp_user = get_user_by('email', $moodle_user['email']);
            
            if ($wp_user) {
                // کاربر WordPress وجود دارد اما Moodle ID متفاوت است
                // به‌روزرسانی Moodle ID
                update_user_meta($wp_user->ID, 'mws_moodle_user_id', intval($moodle_user['id']));
                update_user_meta($wp_user->ID, 'mws_synced_at', current_time('mysql'));
                $updated++;
            } else {
                // ایجاد کاربر جدید در WordPress
                $username = sanitize_user($moodle_user['username'], true);
                
                // اگر username تکراری است، یک عدد اضافه کن
                if (username_exists($username)) {
                    $username = $username . '_' . $moodle_user['id'];
                }
                
                // اگر باز هم تکراری است، از email استفاده کن
                if (username_exists($username)) {
                    $username = sanitize_user($moodle_user['email'], true);
                }
                
                // اگر email هم تکراری است، از ID استفاده کن
                if (email_exists($moodle_user['email'])) {
                    $skipped++;
                    $errors[] = array(
                        'moodle_id' => $moodle_user_id,
                        'username' => $moodle_username,
                        'email' => $moodle_email,
                        'reason' => 'ایمیل تکراری است (کاربر با این ایمیل در WordPress وجود دارد)'
                    );
                    continue;
                }
                
                // آماده‌سازی نام و نام خانوادگی
                $first_name = isset($moodle_user['firstname']) ? trim($moodle_user['firstname']) : '';
                $last_name = isset($moodle_user['lastname']) ? trim($moodle_user['lastname']) : '';
                
                if (empty($first_name) && empty($last_name)) {
                    $first_name = isset($moodle_user['fullname']) ? trim($moodle_user['fullname']) : '';
                    if (empty($first_name)) {
                        $first_name = isset($moodle_user['username']) ? $moodle_user['username'] : 'User';
                    }
                }
                
                $display_name = isset($moodle_user['fullname']) ? trim($moodle_user['fullname']) : '';
                if (empty($display_name)) {
                    $display_name = trim($first_name . ' ' . $last_name);
                    if (empty($display_name)) {
                        $display_name = $username;
                    }
                }
                
                $user_data = array(
                    'user_login' => $username,
                    'user_email' => $moodle_user['email'],
                    'user_pass' => wp_generate_password(12, false),
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                    'display_name' => $display_name,
                    'role' => 'subscriber'
                );
                
                $wp_user_id = wp_insert_user($user_data);
                
                if (!is_wp_error($wp_user_id)) {
                    update_user_meta($wp_user_id, 'mws_moodle_user_id', intval($moodle_user['id']));
                    update_user_meta($wp_user_id, 'mws_synced_at', current_time('mysql'));
                    update_user_meta($wp_user_id, 'mws_imported_from_moodle', true);
                    $imported++;
                } else {
                    $skipped++;
                    $errors[] = array(
                        'moodle_id' => $moodle_user_id,
                        'username' => $moodle_username,
                        'email' => $moodle_email,
                        'reason' => 'خطا در ایجاد کاربر: ' . $wp_user_id->get_error_message()
                    );
                }
            }
        }
        
        return array(
            'success' => true,
            'message' => sprintf('دریافت از Moodle کامل شد. جدید: %d، به‌روزرسانی: %d، رد شده: %d (از %d کاربر دریافت شده)', $imported, $updated, $skipped, count($moodle_users)),
            'imported' => $imported,
            'updated' => $updated,
            'skipped' => $skipped,
            'total_received' => count($moodle_users),
            'errors' => $errors
        );
    }
    
    /**
     * دریافت کاربران از Moodle (AJAX)
     */
    public function ajax_import_users_from_moodle() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        if (!$this->moodle_api->is_configured()) {
            wp_send_json_error(array('message' => 'لطفاً ابتدا Moodle را در تنظیمات پیکربندی کنید'));
        }
        
        $result = $this->import_users_from_moodle();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    /**
     * همگام‌سازی کاربر جدید به Moodle
     */
    public function sync_user_to_moodle($user_id) {
        if (!$this->moodle_api->is_configured()) {
            return false;
        }
        
        $user = get_userdata($user_id);
        if (!$user) {
            return false;
        }
        
        // بررسی اینکه آیا کاربر قبلاً در Moodle وجود دارد (بر اساس email)
        $moodle_user = $this->moodle_api->get_user_by_email($user->user_email);
        
        if ($moodle_user) {
            // به‌روزرسانی اطلاعات کاربر
            update_user_meta($user_id, 'mws_moodle_user_id', $moodle_user['id']);
            $this->log_sync($user_id, $moodle_user['id'], 'update', 'success', null, 'User found in Moodle');
            return true;
        }
        
        // آماده‌سازی username برای Moodle
        // ابتدا بررسی کن که آیا username در Moodle تکراری است
        $base_username = sanitize_user($user->user_login, true);
        if (empty($base_username)) {
            $base_username = sanitize_user($user->user_email, true);
        }
        if (empty($base_username)) {
            $base_username = 'user_' . $user_id;
        }
        
        // بررسی تکراری بودن username در Moodle
        $username = $base_username;
        $counter = 1;
        while ($this->moodle_api->get_user_by_username($username)) {
            $username = $base_username . '_' . $counter;
            $counter++;
            // جلوگیری از حلقه بی‌نهایت
            if ($counter > 100) {
                $username = 'user_' . $user_id . '_' . time();
                break;
            }
        }
        
        // آماده‌سازی firstname و lastname
        $firstname = !empty($user->first_name) ? trim($user->first_name) : '';
        $lastname = !empty($user->last_name) ? trim($user->last_name) : '';
        
        // اگر firstname خالی بود، از display_name استفاده کن
        if (empty($firstname)) {
            $firstname = !empty($user->display_name) ? trim($user->display_name) : 'User';
        }
        
        // اگر firstname هنوز خالی است، از username استفاده کن
        if (empty($firstname)) {
            $firstname = $username;
        }
        
        // Moodle نیاز به حداقل یک کاراکتر برای firstname دارد
        if (strlen($firstname) < 1) {
            $firstname = 'User';
        }
        
        // lastname می‌تواند خالی باشد، اما بهتر است یک مقدار داشته باشد
        if (empty($lastname)) {
            $lastname = '';
        }
        
        $user_data = array(
            'username' => $username,
            'password' => wp_generate_password(12, false),
            'firstname' => $firstname,
            'lastname' => $lastname,
            'email' => $user->user_email
        );
        
        $result = $this->moodle_api->create_user($user_data);
        
        if (!is_wp_error($result) && isset($result[0]['id'])) {
            $moodle_user_id = $result[0]['id'];
            update_user_meta($user_id, 'mws_moodle_user_id', $moodle_user_id);
            update_user_meta($user_id, 'mws_synced_at', current_time('mysql'));
            $this->log_sync($user_id, $moodle_user_id, 'create', 'success', null, 'User created in Moodle');
            return true;
        }
        
        $error_message = is_wp_error($result) ? $result->get_error_message() : 'Failed to create user in Moodle';
        $this->log_sync($user_id, 0, 'create', 'failed', null, $error_message);
        return false;
    }
    
    /**
     * به‌روزرسانی کاربر در Moodle
     */
    public function update_user_in_moodle($user_id, $old_user_data) {
        if (!$this->moodle_api->is_configured()) {
            return false;
        }
        
        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        
        // اگر کاربر در Moodle نیست، آن را ایجاد کن
        if (!$moodle_user_id) {
            return $this->sync_user_to_moodle($user_id);
        }
        
        $user = get_userdata($user_id);
        if (!$user) {
            return false;
        }
        
        // به‌روزرسانی اطلاعات کاربر در Moodle
        $user_data = array(
            'firstname' => !empty($user->first_name) ? $user->first_name : $user->display_name,
            'lastname' => !empty($user->last_name) ? $user->last_name : '',
            'email' => $user->user_email
        );
        
        $result = $this->moodle_api->update_user($moodle_user_id, $user_data);
        
        if (!is_wp_error($result)) {
            update_user_meta($user_id, 'mws_synced_at', current_time('mysql'));
            $this->log_sync($user_id, $moodle_user_id, 'update', 'success', null, 'User updated in Moodle');
            return true;
        }
        
        $error_message = $result->get_error_message();
        $this->log_sync($user_id, $moodle_user_id, 'update', 'failed', null, $error_message);
        return false;
    }
    
    /**
     * ثبت لاگ همگام‌سازی
     */
    private function log_sync($user_id, $moodle_user_id, $action, $status, $course_id = null, $message = '') {
        global $wpdb;
        $table_name = $wpdb->prefix . 'mws_sync_log';
        
        $wpdb->insert(
            $table_name,
            array(
                'user_id' => intval($user_id),
                'moodle_user_id' => $moodle_user_id ? intval($moodle_user_id) : null,
                'course_id' => $course_id ? intval($course_id) : null,
                'action' => sanitize_text_field($action),
                'status' => sanitize_text_field($status),
                'message' => sanitize_text_field($message),
                'created_at' => current_time('mysql')
            ),
            array('%d', '%d', '%d', '%s', '%s', '%s', '%s')
        );
    }
    
    /**
     * همگام‌سازی دستی یک کاربر (AJAX)
     */
    public function ajax_sync_user() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        
        if (!$user_id) {
            wp_send_json_error(array('message' => 'شناسه کاربر نامعتبر است'));
        }
        
        $result = $this->sync_user_to_moodle($user_id);
        
        if ($result) {
            $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
            wp_send_json_success(array(
                'message' => 'همگام‌سازی با موفقیت انجام شد',
                'moodle_user_id' => $moodle_user_id
            ));
        } else {
            wp_send_json_error(array('message' => 'همگام‌سازی با خطا مواجه شد'));
        }
    }
    
    /**
     * آمار همگام‌سازی کاربران (AJAX)
     */
    public function ajax_get_user_sync_stats() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $total = count(get_users(array('fields' => 'ID', 'number' => -1)));
        
        global $wpdb;
        $synced = (int) $wpdb->get_var(
            "SELECT COUNT(DISTINCT user_id) FROM {$wpdb->usermeta} WHERE meta_key = 'mws_moodle_user_id' AND meta_value != '' AND meta_value != '0'"
        );
        
        wp_send_json_success(array(
            'total' => $total,
            'synced' => $synced,
            'remaining' => max(0, $total - $synced)
        ));
    }

    /**
     * همگام‌سازی دسته‌ای کاربران (AJAX)
     */
    public function ajax_sync_all_users() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        if (!$this->moodle_api->is_configured()) {
            wp_send_json_error(array('message' => 'لطفاً ابتدا Moodle را در تنظیمات پیکربندی کنید'));
        }
        
        $batch_size = (int) get_option('mws_sync_batch_size', 100);
        $batch_size = max(10, min(500, $batch_size));
        
        $offset = isset($_POST['offset']) ? max(0, (int) $_POST['offset']) : 0;
        
        $all_user_ids = get_users(array('fields' => 'ID', 'number' => -1, 'orderby' => 'ID', 'order' => 'ASC'));
        $total_users = count($all_user_ids);
        $batch_user_ids = array_slice($all_user_ids, $offset, $batch_size);
        
        $synced = 0;
        $failed = 0;
        $errors = array();
        
        foreach ($batch_user_ids as $user_id) {
            $result = $this->sync_user_to_moodle($user_id);
            if ($result) {
                $synced++;
            } else {
                $failed++;
                global $wpdb;
                $table_name = $wpdb->prefix . 'mws_sync_log';
                $last_error = $wpdb->get_var($wpdb->prepare(
                    "SELECT message FROM $table_name WHERE user_id = %d AND status = 'failed' ORDER BY created_at DESC LIMIT 1",
                    $user_id
                ));
                if ($last_error) {
                    $user = get_userdata($user_id);
                    $errors[] = ($user ? $user->display_name : $user_id) . ': ' . $last_error;
                }
            }
        }
        
        $from = $offset + 1;
        $to = $offset + count($batch_user_ids);
        $next_offset = $offset + count($batch_user_ids);
        $has_more = $next_offset < $total_users;
        
        $message = sprintf('کاربران %d تا %d: موفق %d، ناموفق %d.', $from, $to, $synced, $failed);
        if ($has_more) {
            $message .= ' برای ادامه دکمه را دوباره بزنید.';
        } else {
            $message = 'همگام‌سازی همه کاربران تمام شد. ' . $message;
        }
        if ($failed > 0 && count($errors) > 0) {
            $message .= '<br><strong>خطاها:</strong><br>' . implode('<br>', array_slice($errors, 0, 5));
            if (count($errors) > 5) {
                $message .= '<br>و ' . (count($errors) - 5) . ' خطای دیگر...';
            }
        }
        
        wp_send_json_success(array(
            'message' => $message,
            'synced' => $synced,
            'failed' => $failed,
            'offset' => $offset,
            'next_offset' => $next_offset,
            'total_users' => $total_users,
            'from' => $from,
            'to' => $to,
            'has_more' => $has_more
        ));
    }
    
    /**
     * دریافت لاگ همگام‌سازی کاربر
     */
    public function get_user_sync_log($user_id, $limit = 10) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'mws_sync_log';
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name WHERE user_id = %d ORDER BY created_at DESC LIMIT %d",
            $user_id,
            $limit
        ));
    }
}

