<?php
/**
 * کلاس مدیریت اتصال کاربران به دوره‌ها
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_User_Enrollment_Manager {
    private $moodle_api;
    private $enrollment;
    
    public function __construct() {
        if (class_exists('MWS_Moodle_API')) {
            $this->moodle_api = new MWS_Moodle_API();
        }
        if (class_exists('MWS_Enrollment')) {
            $this->enrollment = new MWS_Enrollment();
        }
    }
    
    /**
     * دریافت instance از Enrollment
     */
    private function get_enrollment() {
        if (!$this->enrollment && class_exists('MWS_Enrollment')) {
            $this->enrollment = new MWS_Enrollment();
        }
        return $this->enrollment;
    }
    
    /**
     * راه‌اندازی هوک‌ها
     */
    public function init() {
        // منوی اتصال کاربران در class-admin اضافه می‌شود
        add_action('wp_ajax_mws_unenroll_user', array($this, 'ajax_unenroll_user'));
        add_action('wp_ajax_mws_enroll_user_manual', array($this, 'ajax_enroll_user_manual'));
        add_action('wp_ajax_mws_search_users', array($this, 'ajax_search_users'));
        add_action('wp_ajax_mws_search_courses', array($this, 'ajax_search_courses'));
        add_action('wp_ajax_mws_update_enrollment_expiry', array($this, 'ajax_update_enrollment_expiry'));
    }
    
    /**
     * جستجوی کاربران (AJAX)
     */
    public function ajax_search_users() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $search = isset($_GET['q']) ? sanitize_text_field($_GET['q']) : '';
        $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
        $per_page = 20;
        
        $args = array(
            'number' => $per_page,
            'offset' => ($page - 1) * $per_page,
            'orderby' => 'display_name',
            'order' => 'ASC'
        );
        
        if (!empty($search)) {
            $args['search'] = '*' . $search . '*';
            $args['search_columns'] = array('user_login', 'user_nicename', 'user_email', 'display_name');
        }
        
        $users = get_users($args);
        
        // Get total count efficiently (only count, don't fetch all)
        $count_args = $args;
        $count_args['number'] = -1;
        $count_args['fields'] = 'ID';
        $count_args['count_total'] = true;
        $user_query = new WP_User_Query($count_args);
        $total = $user_query->get_total();
        
        $results = array();
        foreach ($users as $user) {
            $results[] = array(
                'id' => $user->ID,
                'text' => $user->display_name . ' (' . $user->user_email . ')'
            );
        }
        
        wp_send_json(array(
            'results' => $results,
            'pagination' => array(
                'more' => ($page * $per_page) < $total
            )
        ));
    }
    
    /**
     * جستجوی دوره‌ها (AJAX)
     */
    public function ajax_search_courses() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $search = isset($_GET['q']) ? sanitize_text_field($_GET['q']) : '';
        $page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
        $per_page = 20;
        
        $args = array(
            'post_type' => 'moodle_course',
            'posts_per_page' => $per_page,
            'offset' => ($page - 1) * $per_page,
            'post_status' => 'publish',
            'orderby' => 'title',
            'order' => 'ASC'
        );
        
        if (!empty($search)) {
            $args['s'] = $search;
        }
        
        $courses = get_posts($args);
        
        // Get total count efficiently
        $count_args = $args;
        $count_args['posts_per_page'] = -1;
        $count_args['fields'] = 'ids';
        $count_args['no_found_rows'] = false;
        $count_query = new WP_Query($count_args);
        $total = $count_query->found_posts;
        
        $results = array();
        foreach ($courses as $course) {
            $moodle_id = get_post_meta($course->ID, 'mws_moodle_course_id', true);
            if ($moodle_id) {
                $results[] = array(
                    'id' => $moodle_id,
                    'text' => $course->post_title . ' (ID: ' . $moodle_id . ')'
                );
            }
        }
        
        wp_send_json(array(
            'results' => $results,
            'pagination' => array(
                'more' => ($page * $per_page) < $total
            )
        ));
    }
    
    /**
     * صفحه اتصال کاربران
     */
    public function enrollment_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // پردازش فرم اتصال دستی
        if (isset($_POST['mws_enroll_manual']) && check_admin_referer('mws_enroll_manual')) {
            $user_id = intval($_POST['user_id']);
            $course_id = intval($_POST['course_id']);
            
            if ($user_id && $course_id) {
                $result = $this->enrollment->enroll_user($user_id, $course_id);
                if ($result) {
                    echo '<div class="notice notice-success"><p>کاربر با موفقیت به دوره متصل شد.</p></div>';
                } else {
                    echo '<div class="notice notice-error"><p>خطا در اتصال کاربر به دوره.</p></div>';
                }
            }
        }
        
        // دوره‌ها دیگر در اینجا بارگذاری نمی‌شوند (از AJAX استفاده می‌شود)
        
        // دریافت اتصالات با pagination (بهینه‌سازی شده)
        // استفاده از pagination پیش‌فرض WordPress (20 مورد)
        $per_page = apply_filters('mws_enrollments_per_page', 20);
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $filter_user_id = isset($_GET['filter_user_id']) ? max(0, intval($_GET['filter_user_id'])) : 0;
        
        // اگر Moodle تنظیم شده باشد، از Moodle بخوانیم (اولویت اول)
        if ($this->moodle_api->is_configured()) {
            $enrollment_data = $this->get_enrollments_from_moodle_paginated($current_page, $per_page, $filter_user_id);
            $enrollments = $enrollment_data['data'];
            $total_items = $enrollment_data['total'];
            
            // Fallback: اگر Moodle چیزی برنگرداند، از WP meta بخوانیم
            if (empty($enrollments)) {
                $all_enrollments = $this->get_all_enrollments($filter_user_id);
                $total_items = count($all_enrollments);
                $offset = ($current_page - 1) * $per_page;
                $enrollments = array_slice($all_enrollments, $offset, $per_page);
            }
            
            $total_pages = ceil($total_items / $per_page);
        } else {
            // دریافت از WordPress (بهینه‌سازی شده)
            $all_enrollments = $this->get_all_enrollments($filter_user_id);
            $total_items = count($all_enrollments);
            $total_pages = ceil($total_items / $per_page);
            $offset = ($current_page - 1) * $per_page;
            $enrollments = array_slice($all_enrollments, $offset, $per_page);
        }
        
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline">اتصال کاربران به دوره‌ها</h1>
            <a href="<?php echo admin_url('admin.php?page=royal-moodle-sync-enrollment&action=add'); ?>" class="page-title-action">افزودن اتصال جدید</a>
            <hr class="wp-header-end">
            <div class="mws-enrollment-container" style="display: grid; grid-template-columns: 2fr 1fr; gap: 20px; margin-top: 20px;">
                <div class="mws-enrollment-list">
                    <div class="mws-enrollment-filter" style="margin-bottom: 15px; display: flex; align-items: center; gap: 12px; flex-wrap: wrap;">
                        <label for="mws_filter_user_id" style="font-weight: 600; white-space: nowrap;">جستجوی کاربر:</label>
                        <select id="mws_filter_user_id" class="mws-select2" style="width: 320px; min-width: 200px;" data-placeholder="نام یا ایمیل کاربر را تایپ کنید...">
                            <option value=""><?php echo $filter_user_id ? '' : '-- همه اتصالات --'; ?></option>
                            <?php
                            if ($filter_user_id) {
                                $filter_user = get_userdata($filter_user_id);
                                if ($filter_user) {
                                    echo '<option value="' . esc_attr($filter_user_id) . '" selected="selected">' . esc_html($filter_user->display_name . ' (' . $filter_user->user_email . ')') . '</option>';
                                }
                            }
                            ?>
                        </select>
                        <?php if ($filter_user_id) : ?>
                            <a href="<?php echo esc_url(admin_url('admin.php?page=royal-moodle-sync-enrollment')); ?>" class="button">پاک کردن فیلتر</a>
                        <?php endif; ?>
                    </div>
                    
                    <table class="wp-list-table widefat fixed striped">
                    <thead>
                            <tr>
                                <th>کاربر</th>
                                <th>دوره</th>
                                <th>شناسه Moodle</th>
                                <th>تاریخ اتصال</th>
                                <th>تاریخ پایان دوره</th>
                                <th>نمره</th>
                                <th>Cohort</th>
                                <th>عملیات</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($enrollments)) : ?>
                                <tr>
                                    <td colspan="8">
                                        <p style="padding: 20px; text-align: center; color: #666;">
                                            هیچ اتصالی یافت نشد.
                                            <br>
                                            <small>از فرم سمت راست می‌توانید یک اتصال دستی ایجاد کنید.</small>
                                        </p>
                                    </td>
                                </tr>
                            <?php else : ?>
                                <?php foreach ($enrollments as $enrollment) : 
                                    $user = get_userdata($enrollment['user_id']);
                                    $course = null;
                                    if (isset($enrollment['course_id']) && $enrollment['course_id'] > 0) {
                                        $course = get_post($enrollment['course_id']);
                                    }
                                    $moodle_user_id = get_user_meta($enrollment['user_id'], 'mws_moodle_user_id', true);
                                    
                                    // پیدا کردن دوره Moodle
                                    $moodle_course_id = isset($enrollment['moodle_course_id']) ? $enrollment['moodle_course_id'] : 0;
                                    
                                    // اگر moodle_course_id نداریم اما course_id داریم، از course_id استفاده کنیم
                                    if (!$moodle_course_id && isset($enrollment['course_id']) && $enrollment['course_id'] > 0) {
                                        $moodle_course_id = get_post_meta($enrollment['course_id'], 'mws_moodle_course_id', true);
                                    }
                                    
                                    // دریافت اطلاعات cohort
                                    $cohorts = $this->get_user_cohorts($moodle_user_id);
                                ?>
                                    <tr>
                                        <td>
                                            <?php if ($user) : ?>
                                                <a href="<?php echo admin_url('user-edit.php?user_id=' . $user->ID); ?>">
                                                    <strong><?php echo esc_html($user->display_name); ?></strong>
                                                </a>
                                                <br>
                                                <small><?php echo esc_html($user->user_email); ?></small>
                                            <?php else : ?>
                                                کاربر ID: <?php echo esc_html($enrollment['user_id']); ?>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($course) : ?>
                                                <a href="<?php echo get_edit_post_link($course->ID); ?>">
                                                    <?php echo esc_html($course->post_title); ?>
                                                </a>
                                            <?php else : ?>
                                                <span style="color: #999;">
                                                    دوره Moodle ID: <?php echo esc_html($moodle_course_id); ?>
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <strong><?php echo esc_html($moodle_course_id ?: '-'); ?></strong>
                                        </td>
                                        <td>
                                            <?php echo esc_html($enrollment['enrolled_at']); ?>
                                        </td>
                                        <td class="mws-expiry-cell">
                                            <?php
                                            if (!empty($enrollment['expires_at'])) {
                                                $expired = strtotime($enrollment['expires_at']) < time();
                                                $color   = $expired ? '#d63638' : '#00a32a';
                                                $label   = $expired ? ' (منقضی)' : '';
                                                echo '<span style="color:' . $color . ';">' . esc_html($enrollment['expires_at']) . $label . '</span>';
                                            } else {
                                                echo '<span style="color: #999;">نامحدود</span>';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <?php
                                            $grade_display = $this->get_enrollment_grade_display($moodle_user_id, $moodle_course_id);
                                            if ($grade_display !== null) {
                                                echo '<strong>' . esc_html($grade_display) . '</strong>';
                                            } else {
                                                echo '<span style="color: #999;" title="نمره ثبت نشده یا در دسترس نیست">—</span>';
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <?php if (!empty($cohorts)) : ?>
                                                <?php foreach ($cohorts as $cohort) : ?>
                                                    <span class="mws-cohort-badge" style="background: #2271b1; color: #fff; padding: 2px 8px; border-radius: 3px; font-size: 11px; margin: 2px;">
                                                        <?php echo esc_html($cohort['name']); ?>
                                                    </span>
                                                <?php endforeach; ?>
                                            <?php else : ?>
                                                <span style="color: #999;">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($moodle_course_id) : ?>
                                                <button type="button" 
                                                        class="button button-small mws-unenroll-btn" 
                                                        data-user-id="<?php echo esc_attr($enrollment['user_id']); ?>"
                                                        data-course-id="<?php echo esc_attr($moodle_course_id); ?>"
                                                        data-enrollment-id="<?php echo esc_attr($enrollment['user_id'] . '_' . $moodle_course_id); ?>">
                                                    قطع اتصال
                                                </button>
                                                <button type="button" 
                                                        class="button button-small mws-update-expiry-btn" 
                                                        data-user-id="<?php echo esc_attr($enrollment['user_id']); ?>"
                                                        data-course-id="<?php echo esc_attr($moodle_course_id); ?>"
                                                        data-expires-at="<?php echo esc_attr(!empty($enrollment['expires_at']) ? $enrollment['expires_at'] : ''); ?>"
                                                        style="margin-top: 4px; display: block;">
                                                    تغییر تاریخ پایان
                                                </button>
                                            <?php else : ?>
                                                <span style="color: #999;">-</span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                    
                    <div class="tablenav bottom">
                        <div class="alignleft actions bulkactions">
                            <!-- Bulk actions can be added here if needed -->
                        </div>
                        <div class="tablenav-pages">
                            <span class="displaying-num">
                                <?php echo sprintf(_n('%s مورد', '%s مورد', $total_items, 'royal-moodle-sync'), number_format_i18n($total_items)); ?>
                            </span>
                            <?php if ($total_pages > 1) : ?>
                                <span class="pagination-links">
                                    <?php
                                    $pagination_args = array(
                                        'base' => add_query_arg('paged', '%#%'),
                                        'format' => '',
                                        'prev_text' => '&laquo;',
                                        'next_text' => '&raquo;',
                                        'total' => $total_pages,
                                        'current' => $current_page,
                                        'type' => 'plain'
                                    );
                                    if ($filter_user_id) {
                                        $pagination_args['add_args'] = array('filter_user_id' => $filter_user_id);
                                    }
                                    $page_links = paginate_links($pagination_args);
                                    echo $page_links;
                                    ?>
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <div class="mws-enrollment-form">
                    <div class="postbox" style="margin-top: 0;">
                        <div class="postbox-header">
                            <h2 class="hndle ui-sortable-handle">
                                <span>اتصال دستی کاربر به دوره</span>
                            </h2>
                        </div>
                        <div class="inside" style="padding: 20px;">
                            <form method="post" action="" id="mws-enroll-form">
                                <?php wp_nonce_field('mws_enroll_manual'); ?>
                                
                                <div class="mws-form-group" style="margin-bottom: 20px;">
                                    <label for="user_id" style="display: block; margin-bottom: 8px; font-weight: 600;">
                                        انتخاب کاربر
                                    </label>
                                    <select name="user_id" id="user_id" class="regular-text mws-select2" required style="width: 100%;">
                                        <option value="">-- جستجو و انتخاب کاربر --</option>
                                    </select>
                                    <p class="description" style="margin-top: 5px; color: #646970;">
                                        برای جستجو، شروع به تایپ کنید. می‌توانید بر اساس نام، ایمیل یا نام کاربری جستجو کنید.
                                    </p>
                                </div>
                                
                                <div class="mws-form-group" style="margin-bottom: 20px;">
                                    <label for="course_id" style="display: block; margin-bottom: 8px; font-weight: 600;">
                                        انتخاب دوره
                                    </label>
                                    <select name="course_id" id="course_id" class="regular-text mws-select2" required style="width: 100%;">
                                        <option value="">-- جستجو و انتخاب دوره --</option>
                                    </select>
                                    <p class="description" style="margin-top: 5px; color: #646970;">
                                        برای جستجو، شروع به تایپ کنید. می‌توانید بر اساس عنوان دوره جستجو کنید.
                                    </p>
                                </div>
                                
                                <div class="mws-form-actions" style="margin-top: 25px; padding-top: 20px; border-top: 1px solid #ddd;">
                                    <button type="submit" name="mws_enroll_manual" class="button button-primary button-large" style="width: 100%; padding: 10px; font-size: 14px;">
                                        <span class="dashicons dashicons-yes-alt" style="vertical-align: middle; margin-left: 5px;"></span>
                                        اتصال کاربر به دوره
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Initialize Select2 for searchable dropdowns with AJAX
            if ($.fn.select2) {
                // Select2 for users with AJAX
                $('#user_id').select2({
                    placeholder: 'جستجو و انتخاب کاربر...',
                    allowClear: true,
                    minimumInputLength: 0,
                    ajax: {
                        url: ajaxurl,
                        dataType: 'json',
                        delay: 250,
                        data: function(params) {
                            return {
                                q: params.term || '',
                                page: params.page || 1,
                                action: 'mws_search_users',
                                nonce: '<?php echo wp_create_nonce('mws_ajax_nonce'); ?>'
                            };
                        },
                        processResults: function(data) {
                            return {
                                results: data.results || [],
                                pagination: {
                                    more: data.pagination && data.pagination.more
                                }
                            };
                        },
                        cache: true
                    },
                    language: {
                        noResults: function() {
                            return 'نتیجه‌ای یافت نشد';
                        },
                        searching: function() {
                            return 'در حال جستجو...';
                        },
                        inputTooShort: function() {
                            return 'حداقل 0 کاراکتر وارد کنید';
                        }
                    }
                });
                
                // Select2 for courses with AJAX
                $('#course_id').select2({
                    placeholder: 'جستجو و انتخاب دوره...',
                    allowClear: true,
                    minimumInputLength: 0,
                    ajax: {
                        url: ajaxurl,
                        dataType: 'json',
                        delay: 250,
                        data: function(params) {
                            return {
                                q: params.term || '',
                                page: params.page || 1,
                                action: 'mws_search_courses',
                                nonce: '<?php echo wp_create_nonce('mws_ajax_nonce'); ?>'
                            };
                        },
                        processResults: function(data) {
                            return {
                                results: data.results || [],
                                pagination: {
                                    more: data.pagination && data.pagination.more
                                }
                            };
                        },
                        cache: true
                    },
                    language: {
                        noResults: function() {
                            return 'نتیجه‌ای یافت نشد';
                        },
                        searching: function() {
                            return 'در حال جستجو...';
                        },
                        inputTooShort: function() {
                            return 'حداقل 0 کاراکتر وارد کنید';
                        }
                    }
                });
                
                // Select2 برای فیلتر کاربر (جستجو و مشاهده دوره‌های یک کاربر)
                $('#mws_filter_user_id').select2({
                    placeholder: 'نام یا ایمیل کاربر را تایپ کنید...',
                    allowClear: true,
                    minimumInputLength: 2,
                    ajax: {
                        url: ajaxurl,
                        dataType: 'json',
                        delay: 250,
                        data: function(params) {
                            return {
                                q: params.term || '',
                                page: params.page || 1,
                                action: 'mws_search_users',
                                nonce: '<?php echo wp_create_nonce('mws_ajax_nonce'); ?>'
                            };
                        },
                        processResults: function(data) {
                            return {
                                results: data.results || [],
                                pagination: {
                                    more: data.pagination && data.pagination.more
                                }
                            };
                        },
                        cache: true
                    },
                    language: {
                        noResults: function() { return 'نتیجه‌ای یافت نشد'; },
                        searching: function() { return 'در حال جستجو...'; },
                        inputTooShort: function() { return 'حداقل ۲ کاراکتر وارد کنید'; }
                    }
                });
                $('#mws_filter_user_id').on('select2:select', function(e) {
                    var id = e.params.data.id;
                    if (id) {
                        window.location = '<?php echo esc_url(admin_url('admin.php?page=royal-moodle-sync-enrollment')); ?>&filter_user_id=' + id;
                    }
                });
                $('#mws_filter_user_id').on('select2:clear', function() {
                    window.location = '<?php echo esc_url(admin_url('admin.php?page=royal-moodle-sync-enrollment')); ?>';
                });
            }
            
            var mws_nonce = '<?php echo wp_create_nonce('mws_ajax_nonce'); ?>';
            
            $('.mws-update-expiry-btn').on('click', function() {
                var $btn = $(this);
                var userId = $btn.data('user-id');
                var courseId = $btn.data('course-id');
                var expiresAt = $btn.data('expires-at');
                
                var currentInfo = expiresAt ? 'تاریخ پایان فعلی:\n' + expiresAt : 'تاریخ پایانی تنظیم نشده (نامحدود)';
                var days = prompt(currentInfo + '\n\nتعداد روز را وارد کنید:\n• عدد مثبت (+30) = اضافه کردن روز\n• عدد منفی (-15) = کم کردن روز\n• اگر تاریخ پایانی ندارد، از همین لحظه شروع می‌شود', '30');
                
                if (days === null || days === '') return;
                days = parseInt(days);
                if (isNaN(days) || days === 0) {
                    alert('مقدار وارد شده معتبر نیست. لطفاً یک عدد غیر صفر وارد کنید.');
                    return;
                }
                
                $btn.prop('disabled', true).text('در حال پردازش...');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'mws_update_enrollment_expiry',
                        user_id: userId,
                        course_id: courseId,
                        days: days,
                        nonce: mws_nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            var newExpiresAt = response.data.expires_at || 'نامحدود';
                            $btn.data('expires-at', response.data.expires_at || '');
                            // Update the expires_at cell in the same row
                            var $row = $btn.closest('tr');
                            var $expiryCell = $row.find('.mws-expiry-cell');
                            if (response.data.expires_at) {
                                var now = new Date().getTime();
                                var expTime = new Date(response.data.expires_at.replace(' ', 'T')).getTime();
                                var color = expTime < now ? '#d63638' : '#00a32a';
                                var label = expTime < now ? ' (منقضی)' : '';
                                $expiryCell.html('<span style="color:' + color + ';">' + response.data.expires_at + label + '</span>');
                            } else {
                                $expiryCell.html('<span style="color:#999;">نامحدود</span>');
                            }
                            $btn.prop('disabled', false).text('تغییر تاریخ پایان');
                            alert('موفقیت! تاریخ پایان جدید:\n' + newExpiresAt);
                        } else {
                            alert('خطا: ' + (response.data.message || 'خطا در به‌روزرسانی'));
                            $btn.prop('disabled', false).text('تغییر تاریخ پایان');
                        }
                    },
                    error: function() {
                        alert('خطا در ارتباط با سرور');
                        $btn.prop('disabled', false).text('تغییر تاریخ پایان');
                    }
                });
            });
            
            $('.mws-unenroll-btn').on('click', function() {
                var $btn = $(this);
                var userId = $btn.data('user-id');
                var courseId = $btn.data('course-id');
                var enrollmentId = $btn.data('enrollment-id');
                
                if (!confirm('آیا مطمئن هستید که می‌خواهید این اتصال را قطع کنید؟')) {
                    return;
                }
                
                $btn.prop('disabled', true).text('در حال پردازش...');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'mws_unenroll_user',
                        user_id: userId,
                        course_id: courseId,
                        nonce: '<?php echo wp_create_nonce('mws_ajax_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $btn.closest('tr').fadeOut(function() {
                                $(this).remove();
                            });
                        } else {
                            alert('خطا: ' + (response.data.message || 'خطا در قطع اتصال'));
                            $btn.prop('disabled', false).text('قطع اتصال');
                        }
                    },
                    error: function() {
                        alert('خطا در ارتباط با سرور');
                        $btn.prop('disabled', false).text('قطع اتصال');
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * دریافت همه اتصالات (بهینه‌سازی شده)
     */
    private function get_all_enrollments($filter_user_id = 0) {
        global $wpdb;
        $all_enrollments = array();
        $seen_enrollments = array(); // برای جلوگیری از تکراری
        
        // روش 1: دریافت از WordPress user meta
        $sql = "SELECT user_id, meta_value 
             FROM {$wpdb->usermeta} 
             WHERE meta_key = 'mws_enrollments' 
             AND meta_value != '' 
             AND meta_value != 'a:0:{}'";
        if ($filter_user_id > 0) {
            $sql .= $wpdb->prepare(" AND user_id = %d", $filter_user_id);
        }
        $users_with_enrollments = $wpdb->get_results($sql);
        
        foreach ($users_with_enrollments as $user_meta) {
            $user_id = intval($user_meta->user_id);
            $enrollments = maybe_unserialize($user_meta->meta_value);
            
            if (is_array($enrollments) && !empty($enrollments)) {
                foreach ($enrollments as $enrollment) {
                    if (!isset($enrollment['course_id'])) {
                        continue;
                    }
                    
                    $moodle_course_id = intval($enrollment['course_id']);
                    $enrollment_key = $user_id . '_' . $moodle_course_id;
                    
                    if (isset($seen_enrollments[$enrollment_key])) {
                        continue;
                    }
                    $seen_enrollments[$enrollment_key] = true;
                    
                    $course_id = wp_cache_get('mws_course_' . $moodle_course_id, 'mws_enrollments');
                    if (false === $course_id) {
                        $courses = get_posts(array(
                            'post_type' => 'moodle_course',
                            'meta_query' => array(
                                array(
                                    'key' => 'mws_moodle_course_id',
                                    'value' => $moodle_course_id,
                                    'compare' => '='
                                )
                            ),
                            'posts_per_page' => 1,
                            'fields' => 'ids'
                        ));
                        $course_id = !empty($courses) ? $courses[0] : 0;
                        wp_cache_set('mws_course_' . $moodle_course_id, $course_id, 'mws_enrollments', 3600);
                    }
                    
                    $all_enrollments[] = array(
                        'user_id'         => $user_id,
                        'course_id'       => $course_id,
                        'moodle_course_id'=> $moodle_course_id,
                        'enrolled_at'     => isset($enrollment['enrolled_at']) ? $enrollment['enrolled_at'] : 'نامشخص',
                        'expires_at'      => isset($enrollment['expires_at']) ? $enrollment['expires_at'] : null
                    );
                }
            }
        }
        
        // روش 2: اگر Moodle تنظیم شده باشد و هیچ enrollment در WordPress نبود، از Moodle بخوانیم
        if ($this->moodle_api->is_configured() && empty($all_enrollments)) {
            if ($filter_user_id > 0) {
                $users_with_moodle = $wpdb->get_col($wpdb->prepare(
                    "SELECT user_id FROM {$wpdb->usermeta} 
                     WHERE meta_key = 'mws_moodle_user_id' AND meta_value != '' AND meta_value != '0' AND user_id = %d",
                    $filter_user_id
                ));
            } else {
                $users_with_moodle = $wpdb->get_col(
                    "SELECT DISTINCT user_id 
                     FROM {$wpdb->usermeta} 
                     WHERE meta_key = 'mws_moodle_user_id' 
                     AND meta_value != '' 
                     AND meta_value != '0'
                     LIMIT 50"
                );
            }
            
            foreach ($users_with_moodle as $user_id) {
                $user_id = intval($user_id);
                $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
                
                if ($moodle_user_id) {
                    $moodle_enrollments = $this->moodle_api->get_user_enrollments($moodle_user_id);
                    
                    $wp_meta_enrollments = get_user_meta($user_id, 'mws_enrollments', true);
                    if (!is_array($wp_meta_enrollments)) {
                        $wp_meta_enrollments = array();
                    }
                    $wp_meta_map = array();
                    foreach ($wp_meta_enrollments as $wpe) {
                        if (isset($wpe['course_id'])) {
                            $wp_meta_map[intval($wpe['course_id'])] = $wpe;
                        }
                    }
                    $wp_meta_updated = false;
                    
                    if ($moodle_enrollments && is_array($moodle_enrollments)) {
                        foreach ($moodle_enrollments as $moodle_course) {
                            $moodle_course_id = intval($moodle_course['id']);
                            $enrollment_key = $user_id . '_' . $moodle_course_id;
                            
                            if (isset($seen_enrollments[$enrollment_key])) {
                                continue;
                            }
                            $seen_enrollments[$enrollment_key] = true;
                            
                            $course_id = wp_cache_get('mws_course_' . $moodle_course_id, 'mws_enrollments');
                            if (false === $course_id) {
                                $courses = get_posts(array(
                                    'post_type' => 'moodle_course',
                                    'meta_query' => array(
                                        array(
                                            'key' => 'mws_moodle_course_id',
                                            'value' => $moodle_course_id,
                                            'compare' => '='
                                        )
                                    ),
                                    'posts_per_page' => 1,
                                    'fields' => 'ids'
                                ));
                                $course_id = !empty($courses) ? $courses[0] : 0;
                                wp_cache_set('mws_course_' . $moodle_course_id, $course_id, 'mws_enrollments', 3600);
                            }
                            
                            $enrolled_at = null;
                            $expires_at  = null;
                            
                            if (isset($wp_meta_map[$moodle_course_id])) {
                                $wpe_data    = $wp_meta_map[$moodle_course_id];
                                $enrolled_at = isset($wpe_data['enrolled_at']) ? $wpe_data['enrolled_at'] : null;
                                $expires_at  = isset($wpe_data['expires_at']) ? $wpe_data['expires_at'] : null;
                            }
                            
                            if (empty($enrolled_at)) {
                                if (!empty($moodle_course['lastaccess']) && intval($moodle_course['lastaccess']) > 0) {
                                    $enrolled_at = date('Y-m-d H:i:s', intval($moodle_course['lastaccess']));
                                } else {
                                    $enrolled_at = current_time('mysql');
                                }
                                
                                if (!isset($wp_meta_map[$moodle_course_id])) {
                                    $new_entry = array(
                                        'course_id'   => $moodle_course_id,
                                        'enrolled_at' => $enrolled_at,
                                        'status'      => 'active'
                                    );
                                    $wp_meta_enrollments[]          = $new_entry;
                                    $wp_meta_map[$moodle_course_id] = $new_entry;
                                    $wp_meta_updated = true;
                                }
                            }
                            
                            $all_enrollments[] = array(
                                'user_id'         => $user_id,
                                'course_id'       => $course_id,
                                'moodle_course_id'=> $moodle_course_id,
                                'enrolled_at'     => $enrolled_at,
                                'expires_at'      => $expires_at
                            );
                        }
                        
                        if ($wp_meta_updated) {
                            update_user_meta($user_id, 'mws_enrollments', $wp_meta_enrollments);
                        }
                    }
                }
            }
        }
        
        return $all_enrollments;
    }
    
    /**
     * دریافت اتصالات از Moodle با pagination
     */
    private function get_enrollments_from_moodle_paginated($page = 1, $per_page = 30, $filter_user_id = 0) {
        if (!$this->moodle_api->is_configured()) {
            return array('data' => array(), 'total' => 0);
        }
        
        global $wpdb;
        
        if ($filter_user_id > 0) {
            $users_with_moodle = $wpdb->get_col($wpdb->prepare(
                "SELECT user_id FROM {$wpdb->usermeta} 
                 WHERE meta_key = 'mws_moodle_user_id' AND meta_value != '' AND meta_value != '0' AND user_id = %d",
                $filter_user_id
            ));
        } else {
            $users_with_moodle = $wpdb->get_col(
                "SELECT DISTINCT user_id 
                 FROM {$wpdb->usermeta} 
                 WHERE meta_key = 'mws_moodle_user_id' 
                 AND meta_value != '' 
                 AND meta_value != '0'"
            );
        }
        
        if (empty($users_with_moodle)) {
            return array('data' => array(), 'total' => 0);
        }
        
        $all_enrollments_flat = array();
        $users_to_process = ($filter_user_id > 0) ? $users_with_moodle : array_slice($users_with_moodle, 0, 100);
        
        foreach ($users_to_process as $user_id) {
            $user_id = intval($user_id);
            $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
            
            if ($moodle_user_id) {
                $moodle_enrollments = $this->moodle_api->get_user_enrollments($moodle_user_id);
                
                // بارگذاری اطلاعات ثبت‌نام از WordPress meta برای تاریخ‌ها
                $wp_meta_enrollments = get_user_meta($user_id, 'mws_enrollments', true);
                if (!is_array($wp_meta_enrollments)) {
                    $wp_meta_enrollments = array();
                }
                $wp_meta_map = array();
                foreach ($wp_meta_enrollments as $wpe) {
                    if (isset($wpe['course_id'])) {
                        $wp_meta_map[intval($wpe['course_id'])] = $wpe;
                    }
                }
                $wp_meta_updated = false;
                
                if ($moodle_enrollments && is_array($moodle_enrollments)) {
                    foreach ($moodle_enrollments as $moodle_course) {
                        if (!isset($moodle_course['id'])) {
                            continue;
                        }
                        
                        $moodle_course_id = intval($moodle_course['id']);
                        
                        $course_id = wp_cache_get('mws_course_' . $moodle_course_id, 'mws_enrollments');
                        if (false === $course_id) {
                            $courses = get_posts(array(
                                'post_type' => 'moodle_course',
                                'meta_query' => array(
                                    array(
                                        'key' => 'mws_moodle_course_id',
                                        'value' => $moodle_course_id,
                                        'compare' => '='
                                    )
                                ),
                                'posts_per_page' => 1,
                                'fields' => 'ids'
                            ));
                            $course_id = !empty($courses) ? $courses[0] : 0;
                            wp_cache_set('mws_course_' . $moodle_course_id, $course_id, 'mws_enrollments', 3600);
                        }
                        
                        $enrolled_at = null;
                        $expires_at  = null;
                        
                        if (isset($wp_meta_map[$moodle_course_id])) {
                            $wpe_data    = $wp_meta_map[$moodle_course_id];
                            $enrolled_at = isset($wpe_data['enrolled_at']) ? $wpe_data['enrolled_at'] : null;
                            $expires_at  = isset($wpe_data['expires_at']) ? $wpe_data['expires_at'] : null;
                        }
                        
                        // اگر enrolled_at نداشتیم، از lastaccess مودل استفاده می‌کنیم
                        if (empty($enrolled_at)) {
                            if (!empty($moodle_course['lastaccess']) && intval($moodle_course['lastaccess']) > 0) {
                                $enrolled_at = date('Y-m-d H:i:s', intval($moodle_course['lastaccess']));
                            } else {
                                $enrolled_at = current_time('mysql');
                            }
                            
                            // ذخیره در WP meta برای دفعات بعدی
                            if (!isset($wp_meta_map[$moodle_course_id])) {
                                $new_entry = array(
                                    'course_id'   => $moodle_course_id,
                                    'enrolled_at' => $enrolled_at,
                                    'status'      => 'active'
                                );
                                $wp_meta_enrollments[]              = $new_entry;
                                $wp_meta_map[$moodle_course_id]     = $new_entry;
                                $wp_meta_updated = true;
                            }
                        }
                        
                        $all_enrollments_flat[] = array(
                            'user_id'         => $user_id,
                            'course_id'       => $course_id,
                            'moodle_course_id'=> $moodle_course_id,
                            'enrolled_at'     => $enrolled_at,
                            'expires_at'      => $expires_at
                        );
                    }
                    
                    // ذخیره یک‌باره به WP meta
                    if ($wp_meta_updated) {
                        update_user_meta($user_id, 'mws_enrollments', $wp_meta_enrollments);
                    }
                }
            }
        }
        
        // Pagination
        $total_items = count($all_enrollments_flat);
        $offset = ($page - 1) * $per_page;
        $enrollments_page = array_slice($all_enrollments_flat, $offset, $per_page);
        
        return array(
            'data' => $enrollments_page,
            'total' => $total_items
        );
    }
    
    /**
     * همگام‌سازی اتصالات از Moodle
     */
    private function sync_enrollments_from_moodle() {
        if (!$this->moodle_api->is_configured()) {
            return false;
        }
        
        // بهینه‌سازی: فقط کاربرانی که Moodle ID دارند
        global $wpdb;
        $user_ids = $wpdb->get_col(
            "SELECT user_id 
             FROM {$wpdb->usermeta} 
             WHERE meta_key = 'mws_moodle_user_id' 
             AND meta_value != '' 
             AND meta_value != '0'"
        );
        
        $synced_count = 0;
        
        // پردازش به صورت batch برای جلوگیری از timeout
        $batch_size = 50;
        $batches = array_chunk($user_ids, $batch_size);
        
        foreach ($batches as $batch) {
            foreach ($batch as $user_id) {
            $user_id = intval($user_id);
            $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
            
            if (!$moodle_user_id) {
                // اگر کاربر در Moodle نیست، آن را همگام‌سازی کن
                $user_sync = new MWS_User_Sync();
                $user_sync->sync_user_to_moodle($user_id);
                $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
            }
            
            if ($moodle_user_id) {
                // دریافت اتصالات از Moodle
                $moodle_enrollments = $this->moodle_api->get_user_enrollments($moodle_user_id);
                
                if ($moodle_enrollments && is_array($moodle_enrollments)) {
                    $wp_enrollments = get_user_meta($user_id, 'mws_enrollments', true);
                    if (!is_array($wp_enrollments)) {
                        $wp_enrollments = array();
                    }
                    
                    $existing_course_ids = array();
                    foreach ($wp_enrollments as $wp_enrollment) {
                        $existing_course_ids[] = intval($wp_enrollment['course_id']);
                    }
                    
                    foreach ($moodle_enrollments as $moodle_course) {
                        $course_id = intval($moodle_course['id']);
                        if (!in_array($course_id, $existing_course_ids)) {
                            $wp_enrollments[] = array(
                                'course_id' => $course_id,
                                'enrolled_at' => current_time('mysql'),
                                'status' => 'active'
                            );
                            $synced_count++;
                        }
                    }
                    
                    update_user_meta($user_id, 'mws_enrollments', $wp_enrollments);
                }
            }
            }
        }
        
        return $synced_count;
    }
    
    /**
     * دریافت cohort های کاربر
     */
    private function get_user_cohorts($moodle_user_id) {
        if (!$moodle_user_id || !$this->moodle_api->is_configured()) {
            return array();
        }
        
        // استفاده از API Moodle برای دریافت cohort ها
        // این نیاز به افزودن تابع core_cohort_get_cohorts دارد
        // برای حالا یک آرایه خالی برمی‌گردانیم
        return array();
    }

    /**
     * استخراج نمره نهایی دوره از پاسخ API نمرات مودل
     * @param int $moodle_user_id شناسه کاربر در مودل
     * @param int $moodle_course_id شناسه دوره در مودل
     * @return string|null نمره برای نمایش، یا null اگر نمره‌ای نباشد
     */
    private function get_enrollment_grade_display($moodle_user_id, $moodle_course_id) {
        if (!$moodle_user_id || !$moodle_course_id || !$this->moodle_api->is_configured()) {
            return null;
        }
        
        // روش اول: gradereport_user_get_grade_items (پاسخ ساختاریافته‌تر)
        $result = $this->moodle_api->get_user_grade_items($moodle_user_id, $moodle_course_id);
        if ($result && !is_wp_error($result) && !empty($result['usergrades'])) {
            $gradeitems = isset($result['usergrades'][0]['gradeitems']) ? $result['usergrades'][0]['gradeitems'] : array();
            foreach ($gradeitems as $item) {
                if (isset($item['itemtype']) && $item['itemtype'] === 'course') {
                    $grade = isset($item['gradeformatted']) ? trim($item['gradeformatted']) : '';
                    if ($grade !== '' && $grade !== '-') {
                        return $grade;
                    }
                    return null;
                }
            }
        }
        
        // روش دوم: gradereport_user_get_grades_table (fallback)
        $result = $this->moodle_api->get_user_grades($moodle_user_id, $moodle_course_id);
        if (!$result || is_wp_error($result) || empty($result['tables'])) {
            return null;
        }
        $tabledata = isset($result['tables'][0]['tabledata']) ? $result['tables'][0]['tabledata'] : array();
        
        // آخرین ردیف جدول معمولاً Course total است
        $last_grade = null;
        foreach ($tabledata as $row) {
            $itemname = isset($row['itemname']['content']) ? strip_tags($row['itemname']['content']) : '';
            if (stripos($itemname, 'Course total') !== false || stripos($itemname, 'جمع کل') !== false) {
                if (!empty($row['grade']['content']) && trim($row['grade']['content']) !== '-' && trim($row['grade']['content']) !== '') {
                    return trim(strip_tags($row['grade']['content']));
                }
                return null;
            }
            if (!empty($row['grade']['content']) && trim($row['grade']['content']) !== '-' && trim($row['grade']['content']) !== '') {
                $last_grade = trim(strip_tags($row['grade']['content']));
            }
        }
        
        return $last_grade;
    }
    
    /**
     * قطع اتصال کاربر (AJAX)
     */
    public function ajax_unenroll_user() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        
        if (!$user_id || !$course_id) {
            wp_send_json_error(array('message' => 'پارامترهای نامعتبر'));
        }
        
        $enrollment = $this->get_enrollment();
        if (!$enrollment) {
            wp_send_json_error(array('message' => 'کلاس Enrollment در دسترس نیست'));
            return;
        }
        
        $result = $enrollment->unenroll_user($user_id, $course_id);
        
        if ($result) {
            wp_send_json_success(array('message' => 'اتصال با موفقیت قطع شد'));
        } else {
            wp_send_json_error(array('message' => 'خطا در قطع اتصال'));
        }
    }
    
    /**
     * اتصال دستی کاربر (AJAX)
     */
    public function ajax_enroll_user_manual() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        
        if (!$user_id || !$course_id) {
            wp_send_json_error(array('message' => 'پارامترهای نامعتبر'));
        }
        
        $enrollment = $this->get_enrollment();
        if (!$enrollment) {
            wp_send_json_error(array('message' => 'کلاس Enrollment در دسترس نیست'));
            return;
        }
        
        $result = $enrollment->enroll_user($user_id, $course_id);
        
        if ($result) {
            wp_send_json_success(array('message' => 'کاربر با موفقیت به دوره متصل شد'));
        } else {
            wp_send_json_error(array('message' => 'خطا در اتصال کاربر به دوره'));
        }
    }
    
    /**
     * به‌روزرسانی تاریخ پایان ثبت‌نام (AJAX)
     */
    public function ajax_update_enrollment_expiry() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $user_id   = isset($_POST['user_id'])   ? intval($_POST['user_id'])   : 0;
        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        $days      = isset($_POST['days'])      ? intval($_POST['days'])      : 0;
        
        if (!$user_id || !$course_id || $days === 0) {
            wp_send_json_error(array('message' => 'پارامترهای نامعتبر'));
        }
        
        $enrollments = get_user_meta($user_id, 'mws_enrollments', true);
        if (!is_array($enrollments)) {
            wp_send_json_error(array('message' => 'ثبت‌نامی در WordPress یافت نشد'));
        }
        
        $found       = false;
        $new_timeend = 0;
        $new_expires = null;
        
        foreach ($enrollments as &$enrollment) {
            if (intval($enrollment['course_id']) === $course_id) {
                $found = true;
                
                if (!empty($enrollment['expires_at'])) {
                    $base_time = strtotime($enrollment['expires_at']);
                } else {
                    $base_time = time();
                }
                
                $new_timeend = $base_time + ($days * DAY_IN_SECONDS);
                
                if ($new_timeend > 0) {
                    $new_expires = date('Y-m-d H:i:s', $new_timeend);
                    $enrollment['expires_at'] = $new_expires;
                } else {
                    $new_expires = null;
                    unset($enrollment['expires_at']);
                }
                break;
            }
        }
        unset($enrollment);
        
        if (!$found) {
            wp_send_json_error(array('message' => 'ثبت‌نام برای این دوره یافت نشد'));
        }
        
        update_user_meta($user_id, 'mws_enrollments', $enrollments);
        
        // به‌روزرسانی در Moodle
        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        if ($moodle_user_id && $this->moodle_api->is_configured()) {
            $this->moodle_api->enroll_user($moodle_user_id, $course_id, 5, 0, max(0, $new_timeend));
        }
        
        wp_send_json_success(array(
            'message'    => 'تاریخ پایان با موفقیت به‌روز شد',
            'expires_at' => $new_expires
        ));
    }
}

// کلاس در فایل اصلی پلاگین راه‌اندازی می‌شود

