<?php
/**
 * کلاس SSO (Single Sign-On) برای لاگین خودکار در Moodle
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_SSO {
    private $moodle_api;
    
    public function __construct() {
        $this->moodle_api = new MWS_Moodle_API();
    }
    
    /**
     * راه‌اندازی هوک‌ها
     */
    public function init() {
        // ایجاد لینک SSO هنگام لاگین کاربر
        add_action('wp_login', array($this, 'create_sso_link'), 10, 2);
        
        // افزودن لینک SSO به منوی کاربر
        add_filter('wp_nav_menu_items', array($this, 'add_sso_menu_item'), 10, 2);
        
        // صفحه SSO
        add_action('init', array($this, 'handle_sso_request'));
        
        // افزودن لینک SSO به پروفایل کاربر
        add_action('show_user_profile', array($this, 'add_sso_profile_link'));
        add_action('edit_user_profile', array($this, 'add_sso_profile_link'));
        
        // Endpoint برای Moodle (برای دریافت اطلاعات کاربر از توکن JWT)
        add_action('init', array($this, 'register_sso_endpoint'));
        add_action('template_redirect', array($this, 'handle_sso_endpoint'));
    }
    
    /**
     * ثبت endpoint برای Moodle
     */
    public function register_sso_endpoint() {
        add_rewrite_rule('^mws-sso-verify/?$', 'index.php?mws_sso_verify=1', 'top');
        add_rewrite_tag('%mws_sso_verify%', '([^&]+)');
    }
    
    /**
     * مدیریت endpoint SSO برای Moodle
     */
    public function handle_sso_endpoint() {
        if (get_query_var('mws_sso_verify') == '1') {
            $token = isset($_GET['token']) ? sanitize_text_field($_GET['token']) : '';
            
            if (empty($token)) {
                wp_send_json_error(array('message' => 'Token is required'));
            }
            
            // بررسی توکن JWT
            $payload = $this->validate_jwt_token($token);
            
            if (!$payload) {
                wp_send_json_error(array('message' => 'Invalid token'));
            }
            
            // برگرداندن اطلاعات کاربر
            wp_send_json_success(array(
                'user_id' => isset($payload['moodle_user_id']) ? $payload['moodle_user_id'] : '',
                'email' => isset($payload['email']) ? $payload['email'] : '',
                'wp_user_id' => isset($payload['user_id']) ? $payload['user_id'] : ''
            ));
        }
    }
    
    /**
     * ایجاد لینک SSO هنگام لاگین
     */
    public function create_sso_link($user_login, $user) {
        if (!$this->moodle_api->is_configured()) {
            return;
        }
        
        $moodle_user_id = get_user_meta($user->ID, 'mws_moodle_user_id', true);
        
        // اگر کاربر در Moodle نیست، آن را همگام‌سازی کن
        if (!$moodle_user_id) {
            $user_sync = new MWS_User_Sync();
            $user_sync->sync_user_to_moodle($user->ID);
            $moodle_user_id = get_user_meta($user->ID, 'mws_moodle_user_id', true);
        }
        
        if ($moodle_user_id) {
            // ذخیره لینک SSO در session
            $sso_token = $this->generate_sso_token($user->ID, $moodle_user_id);
            update_user_meta($user->ID, 'mws_sso_token', $sso_token);
            update_user_meta($user->ID, 'mws_sso_token_time', time());
        }
    }
    
    /**
     * تولید توکن SSO با استفاده از JWT (استاندارد RFC 7519)
     */
    private function generate_sso_token($user_id, $moodle_user_id) {
        $user = get_userdata($user_id);
        if (!$user) {
            return false;
        }
        
        // Header
        $header = array(
            'typ' => 'JWT',
            'alg' => 'HS256'
        );
        
        // Payload
        $payload = array(
            'iss' => home_url(), // Issuer
            'sub' => $user->user_email, // Subject (ایمیل کاربر)
            'aud' => get_option('mws_moodle_url', ''), // Audience (Moodle)
            'iat' => time(), // Issued at
            'jti' => wp_generate_password(32, false), // JWT ID
            'user_id' => $user_id,
            'moodle_user_id' => $moodle_user_id,
            'email' => $user->user_email
        );
        
        // بررسی اینکه آیا مدت اعتبار نامحدود است
        $token_unlimited = get_option('mws_sso_token_unlimited', '0');
        if ($token_unlimited == '1') {
            // اگر نامحدود است، exp را تنظیم نمی‌کنیم (یا یک مقدار خیلی بزرگ)
            $payload['exp'] = time() + (10 * 365 * 24 * 60 * 60); // 10 سال (عملاً نامحدود)
        } else {
            // مدت اعتبار از تنظیمات
            $token_expiry = get_option('mws_sso_token_expiry', 3600);
            $payload['exp'] = time() + intval($token_expiry);
        }
        
        // Secret key (باید در تنظیمات ذخیره شود)
        $secret = get_option('mws_sso_secret_key', '');
        if (empty($secret)) {
            $secret = wp_generate_password(64, true, true);
            update_option('mws_sso_secret_key', $secret);
        }
        
        // Encode
        $base64UrlHeader = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode(json_encode($header)));
        $base64UrlPayload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode(json_encode($payload)));
        
        // Signature
        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, $secret, true);
        $base64UrlSignature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        // JWT
        $jwt = $base64UrlHeader . "." . $base64UrlPayload . "." . $base64UrlSignature;
        
        return $jwt;
    }
    
    /**
     * بررسی اعتبار توکن JWT
     */
    private function validate_jwt_token($token) {
        if (empty($token)) {
            return false;
        }
        
        $parts = explode('.', $token);
        if (count($parts) !== 3) {
            return false;
        }
        
        list($base64UrlHeader, $base64UrlPayload, $base64UrlSignature) = $parts;
        
        // Decode payload
        $payload = json_decode(base64_decode(str_replace(['-', '_'], ['+', '/'], $base64UrlPayload)), true);
        
        if (!$payload || !is_array($payload)) {
            return false;
        }
        
        // بررسی انقضا (فقط اگر نامحدود نباشد)
        $token_unlimited = get_option('mws_sso_token_unlimited', '0');
        if ($token_unlimited != '1' && isset($payload['exp']) && $payload['exp'] < time()) {
            return false;
        }
        
        // بررسی signature
        $secret = get_option('mws_sso_secret_key', '');
        if (empty($secret)) {
            return false;
        }
        
        $signature = hash_hmac('sha256', $base64UrlHeader . "." . $base64UrlPayload, $secret, true);
        $base64UrlSignatureCheck = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
        
        if ($base64UrlSignature !== $base64UrlSignatureCheck) {
            return false;
        }
        
        return $payload;
    }
    
    /**
     * افزودن لینک SSO به منوی کاربر
     */
    public function add_sso_menu_item($items, $args) {
        if (!is_user_logged_in()) {
            return $items;
        }
        
        $user_id = get_current_user_id();
        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        
        if (!$moodle_user_id) {
            return $items;
        }
        
        $moodle_url = get_option('mws_moodle_url', '');
        if (empty($moodle_url)) {
            return $items;
        }
        
        $sso_link = add_query_arg(array(
            'mws_sso' => '1',
            'token' => get_user_meta($user_id, 'mws_sso_token', true)
        ), home_url());
        
        $items .= '<li><a href="' . esc_url($sso_link) . '" target="_blank">ورود به Moodle</a></li>';
        
        return $items;
    }
    
    /**
     * مدیریت درخواست SSO
     */
    public function handle_sso_request() {
        if (!isset($_GET['mws_sso']) || $_GET['mws_sso'] != '1') {
            return;
        }
        
        $token = isset($_GET['token']) ? sanitize_text_field($_GET['token']) : '';
        
        $user_id = null;
        $token_valid = false;
        $payload = null;
        
        // اگر توکن ارسالی وجود دارد، از آن برای تشخیص کاربر استفاده می‌کنیم
        if (!empty($token)) {
            // بررسی توکن JWT
            $payload = $this->validate_jwt_token($token);
            
            if ($payload && isset($payload['user_id'])) {
                // استفاده از user_id از توکن (برای پشتیبانی از لینک SSO در صفحه ویرایش کاربر)
                $user_id = intval($payload['user_id']);
                $token_valid = true;
            }
        }
        
        // اگر توکن معتبر نبود یا وجود نداشت، از کاربر فعلی استفاده می‌کنیم
        if (!$user_id || !$token_valid) {
            // بررسی اینکه آیا کاربر لاگین است (برای لینک‌های مستقیم SSO)
            if (!is_user_logged_in()) {
                wp_redirect(wp_login_url());
                exit;
            }
            
            $user_id = get_current_user_id();
            $token_valid = false;
        }
        
        // بررسی وجود کاربر
        $user = get_userdata($user_id);
        if (!$user) {
            wp_die('کاربر یافت نشد.');
        }
        
        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        if (!$moodle_user_id) {
            wp_die('کاربر در Moodle یافت نشد.');
        }
        
        $current_user_email = strtolower(trim($user->user_email));
        
        // اگر توکن معتبر است، بررسی می‌کنیم که متعلق به همین کاربر باشد
        if ($token_valid && $payload) {
            // بررسی ایمیل (استاندارد SSO)
            if (isset($payload['email']) && strtolower(trim($payload['email'])) !== $current_user_email) {
                // اگر ایمیل مطابقت ندارد، توکن نامعتبر است
                $token_valid = false;
            }
        }
        
        // اگر توکن معتبر نیست یا منقضی شده، یک توکن جدید ایجاد می‌کنیم
        if (!$token_valid) {
            $new_token = $this->generate_sso_token($user_id, $moodle_user_id);
            if ($new_token) {
                update_user_meta($user_id, 'mws_sso_token', $new_token);
                update_user_meta($user_id, 'mws_sso_token_time', time());
                $token = $new_token; // استفاده از توکن جدید
                $token_valid = true;
            } else {
                wp_die('خطا در ایجاد توکن SSO.');
            }
        } else {
            // توکن معتبر است، ذخیره می‌کنیم
            update_user_meta($user_id, 'mws_sso_token', $token);
            update_user_meta($user_id, 'mws_sso_token_time', time());
        }
        
        // اگر توکن معتبر نیست، خطا بده
        if (!$token_valid) {
            wp_die('توکن نامعتبر است.');
        }
        
        $moodle_url = get_option('mws_moodle_url', '');
        if (empty($moodle_url)) {
            wp_die('آدرس Moodle تنظیم نشده است.');
        }
        
        // بررسی اینکه آیا course_id در URL است (برای لینک مستقیم به دوره)
        $course_id = isset($_GET['course_id']) ? intval($_GET['course_id']) : 0;
        
        // ایجاد لینک SSO
        // روش 1: استفاده از Web Service برای ایجاد session
        $sso_success = $this->create_sso_via_webservice($user_id);
        
        if ($sso_success) {
            // اگر SSO از طریق Web Service موفق بود
            $redirect_url = rtrim($moodle_url, '/');
            if ($course_id > 0) {
                $redirect_url .= '/course/view.php?id=' . $course_id;
            }
            wp_redirect($redirect_url);
            exit;
        }
        
        // روش 2: ارسال توکن JWT به Moodle
        // استفاده از لینک SSO با توکن JWT
        $sso_url = $this->create_sso_url_with_jwt($user_id, $moodle_user_id, $course_id);
        
        if ($sso_url) {
            wp_redirect($sso_url);
            exit;
        }
        
        // روش 3: استفاده از لینک مستقیم با توکن در URL
        // دریافت توکن فعلی
        $current_token = get_user_meta($user_id, 'mws_sso_token', true);
        if (empty($current_token)) {
            $current_token = $this->generate_sso_token($user_id, $moodle_user_id);
            update_user_meta($user_id, 'mws_sso_token', $current_token);
            update_user_meta($user_id, 'mws_sso_token_time', time());
        }
        
        // ایجاد لینک SSO با توکن JWT
        $sso_url = $this->create_moodle_sso_url($moodle_user_id, $user_id, $course_id, $current_token);
        
        if ($sso_url) {
            wp_redirect($sso_url);
            exit;
        } else {
            // اگر SSO مستقیم ممکن نبود، لینک به صفحه Moodle با توکن
            $moodle_login_url = rtrim($moodle_url, '/') . '/login/index.php';
            if ($course_id > 0) {
                $moodle_login_url = rtrim($moodle_url, '/') . '/course/view.php?id=' . $course_id;
            }
            // افزودن توکن و endpoint به URL
            $verify_url = home_url('/mws-sso-verify/?token=' . urlencode($current_token));
            $moodle_login_url = add_query_arg(array(
                'mws_sso_token' => urlencode($current_token),
                'mws_sso_email' => urlencode($user->user_email),
                'mws_sso_verify_url' => urlencode($verify_url)
            ), $moodle_login_url);
            wp_redirect($moodle_login_url);
            exit;
        }
    }
    
    /**
     * ایجاد لینک SSO برای Moodle با توکن JWT
     */
    private function create_moodle_sso_url($moodle_user_id, $wp_user_id, $course_id = 0, $token = '') {
        $moodle_url = get_option('mws_moodle_url', '');
        if (empty($moodle_url)) {
            return false;
        }
        
        $user = get_userdata($wp_user_id);
        if (!$user) {
            return false;
        }
        
        // ایجاد لینک SSO با توکن JWT
        // این لینک باید به یک endpoint در Moodle یا WordPress ارسال شود
        // که بتواند توکن را پردازش کند و کاربر را لاگین کند
        
        // روش 1: استفاده از endpoint در WordPress که Moodle بتواند به آن درخواست بدهد
        // اما برای حالا، توکن را در URL به Moodle می‌فرستیم
        // Moodle باید یک پلاگین داشته باشد که این توکن را پردازش کند
        
        // لینک به صفحه دوره یا صفحه اصلی Moodle با توکن
        if ($course_id > 0) {
            $moodle_user_url = rtrim($moodle_url, '/') . '/course/view.php?id=' . $course_id;
        } else {
            $moodle_user_url = rtrim($moodle_url, '/') . '/';
        }
        
        // Endpoint در WordPress برای تایید توکن
        $verify_url = home_url('/mws-sso-verify/?token=' . urlencode($token));
        
        // افزودن توکن JWT و ایمیل به URL
        $moodle_user_url = add_query_arg(array(
            'mws_sso_token' => urlencode($token),
            'mws_sso_email' => urlencode($user->user_email),
            'mws_sso_user_id' => $moodle_user_id,
            'mws_sso_verify_url' => urlencode($verify_url)
        ), $moodle_user_url);
        
        return $moodle_user_url;
    }
    
    /**
     * افزودن لینک SSO به پروفایل کاربر
     */
    public function add_sso_profile_link($user) {
        $moodle_user_id = get_user_meta($user->ID, 'mws_moodle_user_id', true);
        $moodle_url = get_option('mws_moodle_url', '');
        
        if (!$moodle_user_id || empty($moodle_url)) {
            return;
        }
        
        // اگر توکن وجود ندارد یا منقضی شده، یک توکن جدید ایجاد کن
        $sso_token = get_user_meta($user->ID, 'mws_sso_token', true);
        $token_time = get_user_meta($user->ID, 'mws_sso_token_time', true);
        
        if (empty($sso_token) || empty($token_time) || (time() - $token_time > 86400)) {
            $sso_token = $this->generate_sso_token($user->ID, $moodle_user_id);
            update_user_meta($user->ID, 'mws_sso_token', $sso_token);
            update_user_meta($user->ID, 'mws_sso_token_time', time());
        }
        
        ?>
        <h3>دسترسی به Moodle</h3>
        <table class="form-table">
            <tr>
                <th>
                    <label>شناسه Moodle</label>
                </th>
                <td>
                    <strong><?php echo esc_html($moodle_user_id); ?></strong>
                </td>
            </tr>
            <tr>
                <th>
                    <label>لینک SSO</label>
                </th>
                <td>
                    <?php
                    $sso_link = add_query_arg(array(
                        'mws_sso' => '1',
                        'token' => $sso_token
                    ), home_url());
                    ?>
                    <a href="<?php echo esc_url($sso_link); ?>" target="_blank" class="button">
                        ورود به Moodle
                    </a>
                    <p class="description">
                        با کلیک روی این دکمه، به صورت خودکار به Moodle وارد می‌شوید.
                    </p>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * ایجاد لینک SSO با استفاده از Web Service
     * این روش نیاز به تنظیمات خاص در Moodle دارد
     */
    public function create_sso_via_webservice($user_id) {
        if (!$this->moodle_api->is_configured()) {
            return false;
        }
        
        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        if (!$moodle_user_id) {
            return false;
        }
        
        // استفاده از Web Service برای ایجاد session
        // Moodle از نسخه 3.9 به بعد از تابع core_auth_create_user_session پشتیبانی می‌کند
        // اما این تابع نیاز به تنظیمات خاص دارد
        
        // روش جایگزین: استفاده از لینک مستقیم با توکن JWT
        // که Moodle بتواند آن را پردازش کند
        
        return false;
    }
    
    /**
     * ایجاد لینک SSO با استفاده از توکن JWT و redirect به Moodle
     * این روش نیاز به یک پلاگین در Moodle دارد که توکن را پردازش کند
     */
    private function create_sso_url_with_jwt($user_id, $moodle_user_id, $course_id = 0) {
        $moodle_url = get_option('mws_moodle_url', '');
        if (empty($moodle_url)) {
            return false;
        }
        
        $user = get_userdata($user_id);
        if (!$user) {
            return false;
        }
        
        // دریافت یا ایجاد توکن JWT
        $token = get_user_meta($user_id, 'mws_sso_token', true);
        if (empty($token)) {
            $token = $this->generate_sso_token($user_id, $moodle_user_id);
            update_user_meta($user_id, 'mws_sso_token', $token);
            update_user_meta($user_id, 'mws_sso_token_time', time());
        }
        
        // ایجاد لینک SSO
        // برای حالا، توکن را در URL به صفحه اصلی Moodle می‌فرستیم
        // Moodle باید یک پلاگین داشته باشد که این توکن را پردازش کند
        
        $redirect_url = rtrim($moodle_url, '/');
        if ($course_id > 0) {
            $redirect_url .= '/course/view.php?id=' . $course_id;
        } else {
            $redirect_url .= '/';
        }
        
        // ایجاد لینک با توکن JWT
        // استفاده از endpoint در WordPress که Moodle بتواند به آن درخواست بدهد
        $verify_url = home_url('/mws-sso-verify/?token=' . urlencode($token));
        
        // ایجاد لینک SSO به فایل mws-sso.php در root Moodle
        $sso_endpoint = rtrim($moodle_url, '/') . '/mws-sso.php';
        
        // ارسال به Moodle با توکن و endpoint
        $sso_url = add_query_arg(array(
            'mws_sso_token' => urlencode($token),
            'mws_sso_email' => urlencode($user->user_email),
            'mws_sso_user_id' => $moodle_user_id,
            'mws_sso_verify_url' => urlencode($verify_url),
            'redirect' => urlencode($redirect_url)
        ), $sso_endpoint);
        
        return $sso_url;
    }
    
    /**
     * ایجاد لینک SSO با استفاده از External Database
     * این روش نیاز به تنظیم External Database در Moodle دارد
     */
    public function create_sso_via_external_db($user_id) {
        $user = get_userdata($user_id);
        if (!$user) {
            return false;
        }
        
        $moodle_url = get_option('mws_moodle_url', '');
        if (empty($moodle_url)) {
            return false;
        }
        
        // اگر Moodle از External Database استفاده می‌کند
        // می‌توانید مستقیماً با username و password لاگین کنید
        // این نیاز به تنظیمات خاص در Moodle دارد
        
        return false;
    }
}

// کلاس در فایل اصلی پلاگین راه‌اندازی می‌شود

