<?php
/**
 * کلاس مدیریت سفارشات دوره‌ها
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_Orders {
    
    public function __construct() {
        // فقط اگر WooCommerce نصب باشد
        if (class_exists('WooCommerce')) {
            $this->init();
        }
    }
    
    /**
     * راه‌اندازی هوک‌ها
     */
    public function init() {
        // منوی سفارشات در class-admin اضافه می‌شود
    }
    
    /**
     * صفحه سفارشات
     */
    public function orders_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // دریافت جزئیات سفارش
        $order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;
        
        if ($order_id) {
            $this->display_order_details($order_id);
            return;
        }
        
        // لیست سفارشات
        $this->display_orders_list();
    }
    
    /**
     * نمایش لیست سفارشات
     */
    private function display_orders_list() {
        global $wpdb;
        
        // دریافت سفارشات مرتبط با دوره‌ها
        $orders = wc_get_orders(array(
            'limit' => -1,
            'orderby' => 'date',
            'order' => 'DESC',
            'status' => array('wc-completed', 'wc-processing', 'wc-pending', 'wc-on-hold', 'wc-cancelled', 'wc-refunded', 'wc-failed')
        ));
        
        // فیلتر کردن سفارشات مرتبط با دوره‌ها
        $course_orders = array();
        foreach ($orders as $order) {
            foreach ($order->get_items() as $item) {
                $product_id = $item->get_product_id();
                $course_id = get_post_meta($product_id, 'mws_linked_course_id', true);
                
                if ($course_id) {
                    $course_orders[] = array(
                        'order' => $order,
                        'course_id' => $course_id,
                        'product_id' => $product_id,
                        'item' => $item
                    );
                    break; // هر سفارش را یک بار نمایش بده
                }
            }
        }
        
        // پاک‌سازی آرایه تکراری
        $unique_orders = array();
        $seen_order_ids = array();
        foreach ($course_orders as $course_order) {
            $order_id = $course_order['order']->get_id();
            if (!in_array($order_id, $seen_order_ids)) {
                $unique_orders[] = $course_order;
                $seen_order_ids[] = $order_id;
            }
        }
        
        // Pagination
        $per_page = 30;
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $total_items = count($unique_orders);
        $total_pages = ceil($total_items / $per_page);
        $offset = ($current_page - 1) * $per_page;
        $unique_orders = array_slice($unique_orders, $offset, $per_page);
        
        ?>
        <div class="wrap">
            <h1>سفارشات دوره‌ها</h1>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>شماره سفارش</th>
                        <th>خریدار</th>
                        <th>دوره</th>
                        <th>نوع دوره</th>
                        <th>مبلغ</th>
                        <th>وضعیت</th>
                        <th>تاریخ سفارش</th>
                        <th>عملیات</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($unique_orders)) : ?>
                        <tr>
                            <td colspan="8">هیچ سفارشی یافت نشد.</td>
                        </tr>
                    <?php else : ?>
                        <?php foreach ($unique_orders as $course_order) : 
                            $order = $course_order['order'];
                            $course_id = $course_order['course_id'];
                            
                            // پیدا کردن دوره
                            $courses = get_posts(array(
                                'post_type' => 'moodle_course',
                                'meta_query' => array(
                                    array(
                                        'key' => 'mws_moodle_course_id',
                                        'value' => $course_id,
                                        'compare' => '='
                                    )
                                ),
                                'posts_per_page' => 1
                            ));
                            
                            $course_name = 'دوره ID: ' . $course_id;
                            if (!empty($courses)) {
                                $course_name = $courses[0]->post_title;
                            }
                            
                            $customer = $order->get_user();
                            $customer_name = $customer ? $customer->display_name : $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
                            $customer_email = $customer ? $customer->user_email : $order->get_billing_email();
                            
                            // نوع دوره
                            $product_id = $course_order['product_id'];
                            $course_type = 'پولی';
                            $course_price = $order->get_total();
                            
                            // بررسی دوره رایگان
                            if ($course_price == 0) {
                                $course_type = 'رایگان';
                            }
                            
                            // وضعیت سفارش
                            $status = $order->get_status();
                            $status_labels = array(
                                'completed' => 'پرداخت شده',
                                'processing' => 'در حال پردازش',
                                'pending' => 'در انتظار پرداخت',
                                'on-hold' => 'در انتظار',
                                'cancelled' => 'لغو شده',
                                'refunded' => 'بازگشت وجه',
                                'failed' => 'ناموفق'
                            );
                            $status_label = isset($status_labels[$status]) ? $status_labels[$status] : $status;
                            $status_colors = array(
                                'completed' => '#00a32a',
                                'processing' => '#2271b1',
                                'pending' => '#dba617',
                                'on-hold' => '#d63638',
                                'cancelled' => '#d63638',
                                'refunded' => '#d63638',
                                'failed' => '#d63638'
                            );
                            $status_color = isset($status_colors[$status]) ? $status_colors[$status] : '#666';
                        ?>
                            <tr>
                                <td>
                                    <strong>#<?php echo esc_html($order->get_order_number()); ?></strong>
                                </td>
                                <td>
                                    <?php if ($customer) : ?>
                                        <a href="<?php echo admin_url('user-edit.php?user_id=' . $customer->ID); ?>">
                                            <?php echo esc_html($customer_name); ?>
                                        </a>
                                    <?php else : ?>
                                        <?php echo esc_html($customer_name); ?>
                                    <?php endif; ?>
                                    <br>
                                    <small><?php echo esc_html($customer_email); ?></small>
                                </td>
                                <td><?php echo esc_html($course_name); ?></td>
                                <td>
                                    <span style="color: <?php echo $course_type == 'رایگان' ? '#00a32a' : '#2271b1'; ?>; font-weight: bold;">
                                        <?php echo esc_html($course_type); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if ($course_type == 'پولی') : ?>
                                        <strong><?php echo wc_price($course_price); ?></strong>
                                    <?php else : ?>
                                        رایگان
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span style="color: <?php echo esc_attr($status_color); ?>; font-weight: bold;">
                                        <?php echo esc_html($status_label); ?>
                                    </span>
                                </td>
                                <td><?php echo esc_html($order->get_date_created()->date_i18n('Y/m/d H:i')); ?></td>
                                <td>
                                    <a href="<?php echo admin_url('admin.php?page=royal-moodle-sync-orders&order_id=' . $order->get_id()); ?>" class="button button-small">
                                        مشاهده جزئیات
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
            
            <?php if ($total_pages > 1) : ?>
                <div class="tablenav bottom">
                    <div class="tablenav-pages">
                        <?php
                        $page_links = paginate_links(array(
                            'base' => add_query_arg('paged', '%#%'),
                            'format' => '',
                            'prev_text' => '&laquo;',
                            'next_text' => '&raquo;',
                            'total' => $total_pages,
                            'current' => $current_page
                        ));
                        echo $page_links;
                        ?>
                    </div>
                    <div class="tablenav-pages" style="float: right; margin-top: 5px;">
                        <span class="displaying-num">
                            نمایش <?php echo $offset + 1; ?> تا <?php echo min($offset + $per_page, $total_items); ?> از <?php echo $total_items; ?> مورد
                        </span>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * نمایش جزئیات سفارش
     */
    private function display_order_details($order_id) {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            echo '<div class="wrap"><p>سفارش یافت نشد.</p></div>';
            return;
        }
        
        $customer = $order->get_user();
        $customer_id = $customer ? $customer->ID : 0;
        
        // دریافت دوره‌های مرتبط
        $course_items = array();
        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();
            $course_id = get_post_meta($product_id, 'mws_linked_course_id', true);
            
            if ($course_id) {
                $courses = get_posts(array(
                    'post_type' => 'moodle_course',
                    'meta_query' => array(
                        array(
                            'key' => 'mws_moodle_course_id',
                            'value' => $course_id,
                            'compare' => '='
                        )
                    ),
                    'posts_per_page' => 1
                ));
                
                $course_name = 'دوره ID: ' . $course_id;
                if (!empty($courses)) {
                    $course_name = $courses[0]->post_title;
                }
                
                $course_items[] = array(
                    'course_id' => $course_id,
                    'course_name' => $course_name,
                    'product_name' => $item->get_name(),
                    'quantity' => $item->get_quantity(),
                    'price' => $item->get_total()
                );
            }
        }
        
        // وضعیت ثبت‌نام
        $enrollments = array();
        if ($customer_id) {
            $user_enrollments = get_user_meta($customer_id, 'mws_enrollments', true);
            if (is_array($user_enrollments)) {
                foreach ($user_enrollments as $enrollment) {
                    $enrollments[] = $enrollment['course_id'];
                }
            }
        }
        
        ?>
        <div class="wrap">
            <h1>جزئیات سفارش #<?php echo esc_html($order->get_order_number()); ?></h1>
            
            <p>
                <a href="<?php echo admin_url('admin.php?page=royal-moodle-sync-orders'); ?>" class="button">
                    ← بازگشت به لیست سفارشات
                </a>
            </p>
            
            <div class="mws-order-details" style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-top: 20px;">
                <div class="mws-order-section">
                    <h2>اطلاعات خریدار</h2>
                    <table class="form-table">
                        <tr>
                            <th>نام:</th>
                            <td>
                                <?php if ($customer) : ?>
                                    <a href="<?php echo admin_url('user-edit.php?user_id=' . $customer->ID); ?>">
                                        <strong><?php echo esc_html($customer->display_name); ?></strong>
                                    </a>
                                <?php else : ?>
                                    <?php echo esc_html($order->get_billing_first_name() . ' ' . $order->get_billing_last_name()); ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <th>ایمیل:</th>
                            <td><?php echo esc_html($customer ? $customer->user_email : $order->get_billing_email()); ?></td>
                        </tr>
                        <tr>
                            <th>تلفن:</th>
                            <td><?php echo esc_html($order->get_billing_phone()); ?></td>
                        </tr>
                        <tr>
                            <th>آدرس:</th>
                            <td>
                                <?php echo esc_html($order->get_billing_address_1()); ?><br>
                                <?php echo esc_html($order->get_billing_city()); ?><br>
                                <?php echo esc_html($order->get_billing_country()); ?>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div class="mws-order-section">
                    <h2>اطلاعات سفارش</h2>
                    <table class="form-table">
                        <tr>
                            <th>شماره سفارش:</th>
                            <td><strong>#<?php echo esc_html($order->get_order_number()); ?></strong></td>
                        </tr>
                        <tr>
                            <th>تاریخ سفارش:</th>
                            <td><?php echo esc_html($order->get_date_created()->date_i18n('Y/m/d H:i:s')); ?></td>
                        </tr>
                        <tr>
                            <th>وضعیت:</th>
                            <td>
                                <?php
                                $status = $order->get_status();
                                $status_labels = array(
                                    'completed' => 'پرداخت شده',
                                    'processing' => 'در حال پردازش',
                                    'pending' => 'در انتظار پرداخت',
                                    'on-hold' => 'در انتظار',
                                    'cancelled' => 'لغو شده',
                                    'refunded' => 'بازگشت وجه',
                                    'failed' => 'ناموفق'
                                );
                                $status_label = isset($status_labels[$status]) ? $status_labels[$status] : $status;
                                $status_colors = array(
                                    'completed' => '#00a32a',
                                    'processing' => '#2271b1',
                                    'pending' => '#dba617',
                                    'on-hold' => '#d63638',
                                    'cancelled' => '#d63638',
                                    'refunded' => '#d63638',
                                    'failed' => '#d63638'
                                );
                                $status_color = isset($status_colors[$status]) ? $status_colors[$status] : '#666';
                                ?>
                                <span style="color: <?php echo esc_attr($status_color); ?>; font-weight: bold; font-size: 14px;">
                                    <?php echo esc_html($status_label); ?>
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th>مبلغ کل:</th>
                            <td><strong><?php echo wc_price($order->get_total()); ?></strong></td>
                        </tr>
                        <tr>
                            <th>روش پرداخت:</th>
                            <td><?php echo esc_html($order->get_payment_method_title()); ?></td>
                        </tr>
                    </table>
                </div>
            </div>
            
            <div class="mws-order-section" style="margin-top: 30px;">
                <h2>دوره‌های سفارش</h2>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th>دوره</th>
                            <th>شناسه Moodle</th>
                            <th>نوع</th>
                            <th>قیمت</th>
                            <th>وضعیت ثبت‌نام</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($course_items)) : ?>
                            <tr>
                                <td colspan="5">هیچ دوره‌ای در این سفارش یافت نشد.</td>
                            </tr>
                        <?php else : ?>
                            <?php foreach ($course_items as $item) : 
                                $is_enrolled = in_array($item['course_id'], $enrollments);
                            ?>
                                <tr>
                                    <td><strong><?php echo esc_html($item['course_name']); ?></strong></td>
                                    <td><?php echo esc_html($item['course_id']); ?></td>
                                    <td>
                                        <?php if ($item['price'] > 0) : ?>
                                            <span style="color: #2271b1; font-weight: bold;">پولی</span>
                                        <?php else : ?>
                                            <span style="color: #00a32a; font-weight: bold;">رایگان</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($item['price'] > 0) : ?>
                                            <?php echo wc_price($item['price']); ?>
                                        <?php else : ?>
                                            رایگان
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($is_enrolled) : ?>
                                            <span style="color: #00a32a; font-weight: bold;">✓ ثبت‌نام شده</span>
                                        <?php else : ?>
                                            <span style="color: #d63638;">✗ ثبت‌نام نشده</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <?php
    }
}

// کلاس در فایل اصلی پلاگین راه‌اندازی می‌شود

