<?php
/**
 * کلاس ارتباط با Moodle Web Service API
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_Moodle_API {
    private $moodle_url;
    private $token;
    private $timeout = 30;
    
    public function __construct() {
        $this->moodle_url = rtrim(get_option('mws_moodle_url', ''), '/');
        $this->token = get_option('mws_moodle_token', '');
    }
    
    /**
     * بررسی اتصال به Moodle
     */
    public function is_configured() {
        return !empty($this->moodle_url) && !empty($this->token);
    }
    
    /**
     * ارسال درخواست به Moodle Web Service
     */
    public function call($function, $params = array()) {
        if (!$this->is_configured()) {
            return new WP_Error('not_configured', 'Moodle URL یا Token تنظیم نشده است.');
        }
        
        $url = $this->moodle_url . '/webservice/rest/server.php';
        
        $request_params = array(
            'wstoken' => $this->token,
            'wsfunction' => $function,
            'moodlewsrestformat' => 'json'
        );
        
        // ادغام پارامترها
        $request_params = array_merge($request_params, $params);
        
        $response = wp_remote_post($url, array(
            'body' => $request_params,
            'timeout' => $this->timeout,
            'sslverify' => true
        ));
        
        if (is_wp_error($response)) {
            error_log('MWS Moodle API Error: ' . $response->get_error_message());
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            return new WP_Error('http_error', 'خطا در ارتباط با Moodle. کد خطا: ' . $response_code);
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        // بررسی خطای Moodle
        if (isset($data['exception'])) {
            $error_message = isset($data['message']) ? $data['message'] : 'خطای نامشخص از Moodle';
            $debug_info = 'Function: ' . $function . ', Params: ' . json_encode($params);
            error_log('MWS Moodle API Error: ' . $error_message . ' | ' . $debug_info);
            return new WP_Error('moodle_error', $error_message);
        }
        
        return $data;
    }
    
    /**
     * ایجاد کاربر در Moodle
     */
    public function create_user($user_data) {
        $params = array(
            'users' => array(
                array(
                    'username' => sanitize_user($user_data['username']),
                    'password' => $user_data['password'],
                    'firstname' => sanitize_text_field($user_data['firstname']),
                    'lastname' => sanitize_text_field($user_data['lastname']),
                    'email' => sanitize_email($user_data['email']),
                    'auth' => 'manual',
                    'preferences' => array(
                        array(
                            'type' => 'auth_forcepasswordchange',
                            'value' => '0'
                        )
                    )
                )
            )
        );
        
        return $this->call('core_user_create_users', $params);
    }
    
    /**
     * به‌روزرسانی کاربر در Moodle
     */
    public function update_user($user_id, $user_data) {
        $params = array(
            'users' => array(
                array(
                    'id' => $user_id,
                    'firstname' => sanitize_text_field($user_data['firstname']),
                    'lastname' => sanitize_text_field($user_data['lastname']),
                    'email' => sanitize_email($user_data['email'])
                )
            )
        );
        
        return $this->call('core_user_update_users', $params);
    }
    
    /**
     * دریافت اطلاعات کاربر بر اساس ایمیل
     */
    public function get_user_by_email($email) {
        $params = array(
            'criteria' => array(
                array(
                    'key' => 'email',
                    'value' => sanitize_email($email)
                )
            )
        );
        
        $result = $this->call('core_user_get_users', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        if ($result && isset($result['users']) && count($result['users']) > 0) {
            return $result['users'][0];
        }
        
        return false;
    }
    
    /**
     * دریافت اطلاعات کاربر بر اساس username
     */
    public function get_user_by_username($username) {
        $params = array(
            'criteria' => array(
                array(
                    'key' => 'username',
                    'value' => sanitize_user($username)
                )
            )
        );
        
        $result = $this->call('core_user_get_users', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        if ($result && isset($result['users']) && count($result['users']) > 0) {
            return $result['users'][0];
        }
        
        return false;
    }
    
    /**
     * دریافت اطلاعات کاربر بر اساس ID
     */
    public function get_user_by_id($user_id) {
        $params = array(
            'criteria' => array(
                array(
                    'key' => 'id',
                    'value' => intval($user_id)
                )
            )
        );
        
        $result = $this->call('core_user_get_users', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        if ($result && isset($result['users']) && count($result['users']) > 0) {
            return $result['users'][0];
        }
        
        return false;
    }
    
    /**
     * دریافت همه کاربران از Moodle
     */
    public function get_all_users($options = array()) {
        // Moodle نیاز به یک criteria معتبر دارد
        // استفاده از criteria معتبر برای دریافت همه کاربران
        // روش 1: با criteria deleted = 0 (کاربران حذف نشده)
        $params = array(
            'criteria' => array(
                array(
                    'key' => 'deleted',
                    'value' => '0'
                )
            )
        );
        
        $result = $this->call('core_user_get_users', $params);
        
        if (is_wp_error($result)) {
            // روش 2: با criteria confirmed = 1 (کاربران تأیید شده)
            $params = array(
                'criteria' => array(
                    array(
                        'key' => 'confirmed',
                        'value' => '1'
                    )
                )
            );
            $result = $this->call('core_user_get_users', $params);
        }
        
        if (is_wp_error($result)) {
            // روش 3: با criteria suspended = 0 (کاربران فعال)
            $params = array(
                'criteria' => array(
                    array(
                        'key' => 'suspended',
                        'value' => '0'
                    )
                )
            );
            $result = $this->call('core_user_get_users', $params);
        }
        
        if (is_wp_error($result)) {
            // روش 4: بدون criteria (آخرین تلاش)
            $params = array();
            $result = $this->call('core_user_get_users', $params);
        }
        
        if (is_wp_error($result)) {
            // ذخیره خطا برای نمایش بهتر
            return new WP_Error('moodle_api_error', $result->get_error_message());
        }
        
        // بررسی ساختار پاسخ
        if ($result && isset($result['users']) && is_array($result['users'])) {
            $users = $result['users'];
            
            // اگر pagination درخواست شده، pagination را خودمان انجام می‌دهیم
            if (!empty($options['limit'])) {
                $offset = isset($options['offset']) ? intval($options['offset']) : 0;
                $limit = intval($options['limit']);
                return array_slice($users, $offset, $limit);
            }
            
            return $users;
        }
        
        // اگر ساختار متفاوت است (مثلاً یک آرایه مستقیم)
        if ($result && is_array($result) && !isset($result['users'])) {
            // ممکن است پاسخ مستقیماً آرایه کاربران باشد
            $users = $result;
            
            // اگر pagination درخواست شده، pagination را خودمان انجام می‌دهیم
            if (!empty($options['limit'])) {
                $offset = isset($options['offset']) ? intval($options['offset']) : 0;
                $limit = intval($options['limit']);
                return array_slice($users, $offset, $limit);
            }
            
            return $users;
        }
        
        // اگر هیچ کاربری پیدا نشد اما خطایی هم نبود
        if ($result && isset($result['users']) && count($result['users']) == 0) {
            return array(); // آرایه خالی
        }
        
        return false;
    }
    
    /**
     * دریافت لیست دوره‌ها
     */
    public function get_courses($options = array()) {
        $params = array();
        
        // در Moodle API، برای دریافت overviewfiles باید از core_course_get_courses_by_field استفاده کنیم
        // یا باید فیلدهای اضافی را درخواست کنیم
        // فعلاً از روش استاندارد استفاده می‌کنیم
        $params['options'] = array(
            'ids' => array()
        );
        
        $result = $this->call('core_course_get_courses', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        return $result;
    }
    
    /**
     * دریافت اطلاعات یک دوره
     */
    public function get_course($course_id) {
        $params = array(
            'options' => array(
                'ids' => array(intval($course_id))
            )
        );
        
        $result = $this->call('core_course_get_courses', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        if ($result && isset($result[0])) {
            return $result[0];
        }
        
        return false;
    }
    
    /**
     * دریافت اطلاعات کامل یک دوره با فیلدهای اضافی
     */
    public function get_course_by_field($field, $value) {
        $params = array(
            'field' => $field,
            'value' => $value
        );
        
        $result = $this->call('core_course_get_courses_by_field', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        if ($result && isset($result['courses']) && isset($result['courses'][0])) {
            return $result['courses'][0];
        }
        
        return false;
    }
    
    /**
     * ثبت‌نام کاربر در دوره
     */
    public function enroll_user($user_id, $course_id, $role_id = 5, $timestart = 0, $timeend = 0) {
        $enrolment = array(
            'roleid'   => intval($role_id), // 5 = student
            'userid'   => intval($user_id),
            'courseid' => intval($course_id)
        );
        
        if ($timestart > 0) {
            $enrolment['timestart'] = intval($timestart);
        }
        if ($timeend > 0) {
            $enrolment['timeend'] = intval($timeend);
        }
        
        $params = array(
            'enrolments' => array($enrolment)
        );
        
        return $this->call('enrol_manual_enrol_users', $params);
    }
    
    /**
     * لغو ثبت‌نام کاربر از دوره
     */
    public function unenroll_user($user_id, $course_id) {
        $params = array(
            'enrolments' => array(
                array(
                    'userid' => intval($user_id),
                    'courseid' => intval($course_id)
                )
            )
        );
        
        return $this->call('enrol_manual_unenrol_users', $params);
    }
    
    /**
     * دریافت لیست ثبت‌نام‌های کاربر
     */
    public function get_user_enrollments($user_id) {
        $params = array(
            'userid' => intval($user_id)
        );
        
        $result = $this->call('core_enrol_get_users_courses', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        return $result;
    }
    
    /**
     * دریافت نمرات کاربر در یک دوره
     */
    public function get_user_grades($user_id, $course_id) {
        $params = array(
            'courseid' => intval($course_id),
            'userid' => intval($user_id)
        );
        
        $result = $this->call('gradereport_user_get_grades_table', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        return $result;
    }
    
    /**
     * دریافت لیست دانشجویان یک دوره (شامل تعلیق‌شده‌ها)
     */
    public function get_course_enrolled_users($course_id) {
        $params = array(
            'courseid' => intval($course_id),
            'options'  => array(
                array('name' => 'onlyactive',    'value' => 0),
                array('name' => 'userfields',    'value' => 'id,fullname,firstname,lastname,email,suspended,lastcourseaccess'),
                array('name' => 'limitnumber',   'value' => 500),
            )
        );

        $result = $this->call('core_enrol_get_enrolled_users', $params);

        if (is_wp_error($result)) {
            return false;
        }

        return $result;
    }

    /**
     * دریافت آیتم‌های نمره کاربر در یک دوره (API تمیزتر)
     */
    public function get_user_grade_items($user_id, $course_id) {
        $params = array(
            'courseid' => intval($course_id),
            'userid'   => intval($user_id)
        );
        
        $result = $this->call('gradereport_user_get_grade_items', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        return $result;
    }
    
    /**
     * دریافت وضعیت پیشرفت دوره برای کاربر
     */
    public function get_course_completion_status($user_id, $course_id) {
        $params = array(
            'userid' => intval($user_id),
            'courseid' => intval($course_id)
        );
        
        $result = $this->call('core_completion_get_course_completion_status', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        return $result;
    }
    
    /**
     * ایجاد دوره در Moodle
     */
    public function create_course($course_data) {
        $params = array(
            'courses' => array(
                array(
                    'fullname' => sanitize_text_field($course_data['fullname']),
                    'shortname' => sanitize_text_field($course_data['shortname']),
                    'categoryid' => intval($course_data['categoryid']),
                    'summary' => wp_kses_post($course_data['summary']),
                    'visible' => isset($course_data['visible']) ? intval($course_data['visible']) : 1
                )
            )
        );
        
        return $this->call('core_course_create_courses', $params);
    }
    
    /**
     * به‌روزرسانی دوره در Moodle
     */
    public function update_course($course_data) {
        $params = array(
            'courses' => array(
                array(
                    'id' => intval($course_data['id']),
                    'fullname' => sanitize_text_field($course_data['fullname']),
                    'shortname' => sanitize_text_field($course_data['shortname']),
                    'categoryid' => intval($course_data['categoryid']),
                    'summary' => wp_kses_post($course_data['summary']),
                    'visible' => isset($course_data['visible']) ? intval($course_data['visible']) : 1
                )
            )
        );
        
        return $this->call('core_course_update_courses', $params);
    }
    
    /**
     * دریافت لیست دسته‌بندی‌ها
     */
    public function get_categories($options = array()) {
        $params = array();
        
        if (!empty($options['ids'])) {
            $params['criteria'] = array();
            foreach ($options['ids'] as $id) {
                $params['criteria'][] = array(
                    'key' => 'id',
                    'value' => intval($id)
                );
            }
        }
        
        $result = $this->call('core_course_get_categories', $params);
        
        if (is_wp_error($result)) {
            return false;
        }
        
        return $result;
    }
    
    /**
     * ایجاد دسته‌بندی در Moodle
     */
    public function create_category($category_data) {
        $params = array(
            'categories' => array(
                array(
                    'name' => sanitize_text_field($category_data['name']),
                    'parent' => isset($category_data['parent']) ? intval($category_data['parent']) : 0,
                    'description' => isset($category_data['description']) ? wp_kses_post($category_data['description']) : ''
                )
            )
        );
        
        return $this->call('core_course_create_categories', $params);
    }
    
    /**
     * به‌روزرسانی دسته‌بندی در Moodle
     */
    public function update_category($category_data) {
        $params = array(
            'categories' => array(
                array(
                    'id' => intval($category_data['id']),
                    'name' => sanitize_text_field($category_data['name']),
                    'parent' => isset($category_data['parent']) ? intval($category_data['parent']) : 0,
                    'description' => isset($category_data['description']) ? wp_kses_post($category_data['description']) : ''
                )
            )
        );
        
        return $this->call('core_course_update_categories', $params);
    }
    
    /**
     * تست اتصال به Moodle
     */
    public function test_connection() {
        // استفاده از تابع core_course_get_courses که معمولاً همیشه فعال است
        // این تابع نیاز به پارامتر خاصی ندارد و فقط لیست دوره‌ها را برمی‌گرداند
        $result = $this->call('core_course_get_courses', array());
        
        if (is_wp_error($result)) {
            $error_message = $result->get_error_message();
            
            // اگر خطای Access control بود، پیام بهتری نمایش بده
            if (strpos($error_message, 'Access control') !== false || strpos($error_message, 'access') !== false) {
                return array(
                    'success' => false,
                    'message' => 'خطای دسترسی: لطفاً مطمئن شوید که تابع "core_course_get_courses" در Web Service شما فعال است. به Site administration > Plugins > Web services > External services بروید و این تابع را به سرویس خود اضافه کنید.'
                );
            }
            
            return array(
                'success' => false,
                'message' => 'خطا در اتصال: ' . $error_message
            );
        }
        
        // اگر نتیجه آرایه است (لیست دوره‌ها)
        if (is_array($result)) {
            $courses_count = count($result);
            return array(
                'success' => true,
                'message' => sprintf('اتصال موفق! تعداد دوره‌ها: %d', $courses_count),
                'data' => array(
                    'courses_count' => $courses_count,
                    'courses' => $result
                )
            );
        }
        
        // امتحان تابع دیگر
        $result2 = $this->call('core_webservice_get_site_info', array());
        
        if (!is_wp_error($result2) && isset($result2['sitename'])) {
            return array(
                'success' => true,
                'message' => 'اتصال موفق! نام سایت: ' . $result2['sitename'],
                'data' => $result2
            );
        }
        
        return array(
            'success' => false,
            'message' => 'پاسخ نامعتبر از Moodle. لطفاً URL و Token را بررسی کنید.'
        );
    }
}

