<?php
/**
 * کلاس مدیریت ثبت‌نام کاربران در دوره‌های Moodle
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_Enrollment {
    private $moodle_api;
    
    public function __construct() {
        $this->moodle_api = new MWS_Moodle_API();
    }
    
    /**
     * راه‌اندازی هوک‌ها
     */
    public function init() {
        // ثبت‌نام خودکار هنگام خرید از WooCommerce
        if (class_exists('WooCommerce')) {
            add_action('woocommerce_order_status_completed', array($this, 'enroll_on_purchase'));
            add_action('woocommerce_payment_complete', array($this, 'enroll_on_purchase'));
        }
        
        // ثبت‌نام دستی از URL
        add_action('init', array($this, 'handle_enrollment_request'));
        
        // شورت‌کد دکمه ثبت‌نام
        add_shortcode('royal_moodle_enroll_button', array($this, 'enroll_button_shortcode'));
        add_shortcode('moodle_enroll_button', array($this, 'enroll_button_shortcode')); // برای سازگاری با نسخه قبلی
        
        // نمایش دوره‌های ثبت‌نام شده کاربر
        add_shortcode('royal_moodle_my_courses', array($this, 'my_courses_shortcode'));
        add_shortcode('moodle_my_courses', array($this, 'my_courses_shortcode')); // برای سازگاری با نسخه قبلی
        
        // شورتکد نمایش دوره‌های کاربر با گرید و پیشرفت
        add_shortcode('royal_moodle_user_courses', array($this, 'user_courses_grid_shortcode'));
        add_shortcode('moodle_user_courses', array($this, 'user_courses_grid_shortcode')); // برای سازگاری با نسخه قبلی
        
        // شورتکد نمایش تعداد دوره‌های ثبت‌نام شده کاربر
        add_shortcode('royal_moodle_enrolled_count', array($this, 'enrolled_count_shortcode'));
        add_shortcode('moodle_enrolled_count', array($this, 'enrolled_count_shortcode')); // برای سازگاری با نسخه قبلی
        
        // AJAX برای ثبت‌نام
        add_action('wp_ajax_mws_enroll', array($this, 'ajax_enroll'));
        add_action('wp_ajax_mws_unenroll', array($this, 'ajax_unenroll'));
    }
    
    /**
     * ثبت‌نام هنگام خرید از WooCommerce
     */
    public function enroll_on_purchase($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }
        
        $user_id = $order->get_user_id();
        if (!$user_id) {
            return;
        }
        
        // اطمینان از همگام‌سازی کاربر
        $user_sync = new MWS_User_Sync();
        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        if (!$moodle_user_id) {
            $user_sync->sync_user_to_moodle($user_id);
            $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        }
        
        if (!$moodle_user_id) {
            error_log('MWS: Cannot enroll user - Moodle user ID not found');
            return;
        }
        
        // دریافت دوره‌های مرتبط با محصولات سفارش
        foreach ($order->get_items() as $item) {
            $product_id = $item->get_product_id();
            $moodle_course_id = get_post_meta($product_id, 'mws_linked_course_id', true);
            
            if ($moodle_course_id) {
                $this->enroll_user($user_id, intval($moodle_course_id));
            }
        }
    }
    
    /**
     * ثبت‌نام کاربر در دوره
     */
    public function enroll_user($user_id, $course_id) {
        if (!$this->moodle_api->is_configured()) {
            return false;
        }
        
        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        
        // اگر کاربر در Moodle نیست، آن را همگام‌سازی کن
        if (!$moodle_user_id) {
            $user_sync = new MWS_User_Sync();
            $user_sync->sync_user_to_moodle($user_id);
            $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        }
        
        if (!$moodle_user_id) {
            $this->log_enrollment($user_id, $course_id, 'enroll', 'failed', 'Moodle user ID not found');
            return false;
        }
        
        // بررسی اینکه آیا قبلاً ثبت‌نام کرده است
        if ($this->is_user_enrolled($user_id, $course_id)) {
            return true; // قبلاً ثبت‌نام کرده
        }
        
        // محاسبه تاریخ شروع و پایان بر اساس مدت دسترسی دوره
        $timestart = time();
        $timeend   = 0;
        $expires_at = null;
        
        $wp_courses = get_posts(array(
            'post_type'      => 'moodle_course',
            'posts_per_page' => 1,
            'fields'         => 'ids',
            'meta_query'     => array(
                array(
                    'key'     => 'mws_moodle_course_id',
                    'value'   => $course_id,
                    'compare' => '='
                )
            )
        ));
        
        if (!empty($wp_courses)) {
            $duration = intval(get_post_meta($wp_courses[0], 'mws_course_duration_days', true));
            if ($duration > 0) {
                $timeend    = $timestart + ($duration * DAY_IN_SECONDS);
                $expires_at = date('Y-m-d H:i:s', $timeend);
            }
        }
        
        // ثبت‌نام در Moodle
        $result = $this->moodle_api->enroll_user($moodle_user_id, $course_id, 5, $timestart, $timeend);
        
        if (is_wp_error($result)) {
            $this->log_enrollment($user_id, $course_id, 'enroll', 'failed', $result->get_error_message());
            return false;
        }
        
        // ذخیره اطلاعات ثبت‌نام در WordPress
        $enrollments = get_user_meta($user_id, 'mws_enrollments', true);
        if (!is_array($enrollments)) {
            $enrollments = array();
        }
        
        // بررسی تکراری نبودن
        $already_exists = false;
        foreach ($enrollments as $enrollment) {
            if ($enrollment['course_id'] == $course_id) {
                $already_exists = true;
                break;
            }
        }
        
        if (!$already_exists) {
            $new_enrollment = array(
                'course_id'   => intval($course_id),
                'enrolled_at' => current_time('mysql'),
                'status'      => 'active'
            );
            if ($expires_at !== null) {
                $new_enrollment['expires_at'] = $expires_at;
            }
            $enrollments[] = $new_enrollment;
            update_user_meta($user_id, 'mws_enrollments', $enrollments);
        }
        
        $this->log_enrollment($user_id, $course_id, 'enroll', 'success', 'User enrolled successfully');
        return true;
    }
    
    /**
     * لغو ثبت‌نام کاربر از دوره
     */
    public function unenroll_user($user_id, $course_id) {
        if (!$this->moodle_api->is_configured()) {
            return false;
        }
        
        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        
        if (!$moodle_user_id) {
            $this->log_enrollment($user_id, $course_id, 'unenroll', 'failed', 'Moodle user ID not found');
            return false;
        }
        
        // لغو ثبت‌نام در Moodle
        $result = $this->moodle_api->unenroll_user($moodle_user_id, $course_id);
        
        if (is_wp_error($result)) {
            $this->log_enrollment($user_id, $course_id, 'unenroll', 'failed', $result->get_error_message());
            return false;
        }
        
        // حذف از لیست ثبت‌نام‌ها در WordPress
        $enrollments = get_user_meta($user_id, 'mws_enrollments', true);
        if (is_array($enrollments)) {
            $enrollments = array_filter($enrollments, function($enrollment) use ($course_id) {
                return $enrollment['course_id'] != $course_id;
            });
            update_user_meta($user_id, 'mws_enrollments', array_values($enrollments));
        }
        
        $this->log_enrollment($user_id, $course_id, 'unenroll', 'success', 'User unenrolled successfully');
        return true;
    }
    
    /**
     * بررسی اینکه آیا کاربر در دوره ثبت‌نام کرده است
     */
    public function is_user_enrolled($user_id, $course_id) {
        $enrollments = get_user_meta($user_id, 'mws_enrollments', true);
        
        if (!is_array($enrollments)) {
            return false;
        }
        
        foreach ($enrollments as $enrollment) {
            if ($enrollment['course_id'] == $course_id && 
                (!isset($enrollment['status']) || $enrollment['status'] == 'active')) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * دریافت لیست دوره‌های ثبت‌نام شده کاربر
     */
    public function get_user_enrollments($user_id) {
        $enrollments = get_user_meta($user_id, 'mws_enrollments', true);
        
        if (!is_array($enrollments)) {
            return array();
        }
        
        return $enrollments;
    }
    
    /**
     * مدیریت درخواست ثبت‌نام از URL
     */
    public function handle_enrollment_request() {
        if (!is_user_logged_in()) {
            return;
        }
        
        $user_id = get_current_user_id();
        
        // ثبت‌نام
        if (isset($_GET['enroll'])) {
            $course_id = intval($_GET['enroll']);
            
            if ($this->enroll_user($user_id, $course_id)) {
                wp_redirect(add_query_arg('enrolled', '1', remove_query_arg('enroll')));
                exit;
            } else {
                wp_redirect(add_query_arg('enroll_error', '1', remove_query_arg('enroll')));
                exit;
            }
        }
        
        // لغو ثبت‌نام
        if (isset($_GET['unenroll'])) {
            $course_id = intval($_GET['unenroll']);
            
            if ($this->unenroll_user($user_id, $course_id)) {
                wp_redirect(add_query_arg('unenrolled', '1', remove_query_arg('unenroll')));
                exit;
            } else {
                wp_redirect(add_query_arg('unenroll_error', '1', remove_query_arg('unenroll')));
                exit;
            }
        }
    }
    
    /**
     * شورت‌کد دکمه ثبت‌نام
     */
    public function enroll_button_shortcode($atts) {
        if (!is_user_logged_in()) {
            return '<div class="mws-enroll-notice"><p>لطفاً ابتدا <a href="' . esc_url(wp_login_url(get_permalink())) . '">وارد شوید</a> تا بتوانید در دوره ثبت‌نام کنید.</p></div>';
        }
        
        $atts = shortcode_atts(array(
            'course_id' => '',
            'text' => 'ثبت‌نام در دوره',
            'unenroll_text' => 'لغو ثبت‌نام'
        ), $atts);
        
        if (empty($atts['course_id'])) {
            return '<p>شناسه دوره مشخص نشده است.</p>';
        }
        
        $course_id = intval($atts['course_id']);
        $user_id = get_current_user_id();
        $is_enrolled = $this->is_user_enrolled($user_id, $course_id);
        
        ob_start();
        ?>
        <div class="mws-enroll-button-wrapper">
            <?php if ($is_enrolled) : ?>
                <a href="<?php echo esc_url(add_query_arg('unenroll', $course_id)); ?>" class="mws-btn mws-btn-unenroll">
                    <?php echo esc_html($atts['unenroll_text']); ?>
                </a>
                <span class="mws-enrolled-badge">✓ شما در این دوره ثبت‌نام کرده‌اید</span>
            <?php else : ?>
                <a href="<?php echo esc_url(add_query_arg('enroll', $course_id)); ?>" class="mws-btn mws-btn-enroll">
                    <?php echo esc_html($atts['text']); ?>
                </a>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * شورت‌کد نمایش دوره‌های من
     */
    public function my_courses_shortcode($atts) {
        if (!is_user_logged_in()) {
            return '<p>لطفاً ابتدا وارد شوید.</p>';
        }
        
        $user_id = get_current_user_id();
        $enrollments = $this->get_user_enrollments($user_id);
        
        if (empty($enrollments)) {
            return '<p>شما در هیچ دوره‌ای ثبت‌نام نکرده‌اید.</p>';
        }
        
        $course_ids = array();
        foreach ($enrollments as $enrollment) {
            $course_ids[] = $enrollment['course_id'];
        }
        
        $args = array(
            'post_type' => 'moodle_course',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => 'mws_moodle_course_id',
                    'value' => $course_ids,
                    'compare' => 'IN'
                )
            )
        );
        
        $courses = get_posts($args);
        
        if (empty($courses)) {
            return '<p>دوره‌های شما یافت نشد.</p>';
        }
        
        ob_start();
        ?>
        <div class="mws-my-courses">
            <h3>دوره‌های من</h3>
            <ul class="mws-my-courses-list">
                <?php foreach ($courses as $course) : 
                    $moodle_id = get_post_meta($course->ID, 'mws_moodle_course_id', true);
                ?>
                    <li>
                        <a href="<?php echo esc_url(get_permalink($course->ID)); ?>">
                            <?php echo esc_html($course->post_title); ?>
                        </a>
                        <a href="<?php echo esc_url(add_query_arg('unenroll', $moodle_id)); ?>" class="mws-unenroll-link">
                            لغو ثبت‌نام
                        </a>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * ثبت‌نام با AJAX
     */
    public function ajax_enroll() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!is_user_logged_in()) {
            wp_send_json_error(array('message' => 'لطفاً ابتدا وارد شوید'));
        }
        
        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        
        if (!$course_id) {
            wp_send_json_error(array('message' => 'شناسه دوره نامعتبر است'));
        }
        
        $user_id = get_current_user_id();
        $result = $this->enroll_user($user_id, $course_id);
        
        if ($result) {
            wp_send_json_success(array('message' => 'ثبت‌نام با موفقیت انجام شد'));
        } else {
            wp_send_json_error(array('message' => 'ثبت‌نام با خطا مواجه شد'));
        }
    }
    
    /**
     * لغو ثبت‌نام با AJAX
     */
    public function ajax_unenroll() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!is_user_logged_in()) {
            wp_send_json_error(array('message' => 'لطفاً ابتدا وارد شوید'));
        }
        
        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        
        if (!$course_id) {
            wp_send_json_error(array('message' => 'شناسه دوره نامعتبر است'));
        }
        
        $user_id = get_current_user_id();
        $result = $this->unenroll_user($user_id, $course_id);
        
        if ($result) {
            wp_send_json_success(array('message' => 'لغو ثبت‌نام با موفقیت انجام شد'));
        } else {
            wp_send_json_error(array('message' => 'لغو ثبت‌نام با خطا مواجه شد'));
        }
    }
    
    /**
     * شورتکد نمایش دوره‌های کاربر با گرید و پیشرفت
     */
    public function user_courses_grid_shortcode($atts) {
        if (!is_user_logged_in()) {
            return '<div class="mws-enroll-notice"><p>لطفاً ابتدا <a href="' . esc_url(wp_login_url(get_permalink())) . '">وارد شوید</a> تا دوره‌های خود را مشاهده کنید.</p></div>';
        }
        
        $user_id = get_current_user_id();
        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        
        if (!$moodle_user_id) {
            return '<div class="mws-enroll-notice"><p>حساب کاربری شما با Moodle همگام‌سازی نشده است. لطفاً با مدیر تماس بگیرید.</p></div>';
        }
        
        // دریافت دوره‌های کاربر از Moodle
        $moodle_courses = $this->moodle_api->get_user_enrollments($moodle_user_id);
        
        if (!$moodle_courses || !is_array($moodle_courses) || empty($moodle_courses)) {
            return '<div class="mws-enroll-notice"><p>شما در هیچ دوره‌ای ثبت‌نام نکرده‌اید.</p></div>';
        }
        
        $moodle_url = rtrim(get_option('mws_moodle_url', ''), '/');
        
        // اطمینان از وجود توکن SSO
        $sso_token = get_user_meta($user_id, 'mws_sso_token', true);
        $token_time = get_user_meta($user_id, 'mws_sso_token_time', true);
        
        if (empty($sso_token) || empty($token_time) || (time() - $token_time > 86400)) {
            // ایجاد توکن جدید
            $sso_token = $this->generate_sso_token_for_user($user_id, $moodle_user_id);
            update_user_meta($user_id, 'mws_sso_token', $sso_token);
            update_user_meta($user_id, 'mws_sso_token_time', time());
        }
        
        ob_start();
        ?>
        <div class="mws-user-courses-grid">
            <?php foreach ($moodle_courses as $moodle_course) : 
                $course_id = isset($moodle_course['id']) ? intval($moodle_course['id']) : 0;
                $course_name = isset($moodle_course['fullname']) ? $moodle_course['fullname'] : '';
                $course_shortname = isset($moodle_course['shortname']) ? $moodle_course['shortname'] : '';
                $course_summary = isset($moodle_course['summary']) ? $moodle_course['summary'] : '';
                
                // ایجاد لینک SSO برای دوره
                $course_url = add_query_arg(array(
                    'mws_sso' => '1',
                    'token' => $sso_token,
                    'course_id' => $course_id
                ), home_url());
                
                // دریافت تصویر دوره از WordPress (اگر سینک شده باشد)
                $wp_course = get_posts(array(
                    'post_type' => 'moodle_course',
                    'meta_query' => array(
                        array(
                            'key' => 'mws_moodle_course_id',
                            'value' => $course_id,
                            'compare' => '='
                        )
                    ),
                    'posts_per_page' => 1
                ));
                
                $thumbnail_url = '';
                if (!empty($wp_course)) {
                    $thumbnail_id = get_post_thumbnail_id($wp_course[0]->ID);
                    if ($thumbnail_id) {
                        $thumbnail_url = wp_get_attachment_image_url($thumbnail_id, 'medium');
                    }
                }
                
                // اگر تصویر نداشت، از overviewfiles در Moodle استفاده کن
                if (empty($thumbnail_url) && isset($moodle_course['overviewfiles']) && !empty($moodle_course['overviewfiles'])) {
                    $file = $moodle_course['overviewfiles'][0];
                    if (isset($file['fileurl'])) {
                        $thumbnail_url = $file['fileurl'];
                    }
                }
                
                // تصویر پیش‌فرض
                if (empty($thumbnail_url)) {
                    // استفاده از placeholder یا تصویر پیش‌فرض
                    $thumbnail_url = 'https://via.placeholder.com/400x200/e0e0e0/999999?text=' . urlencode('بدون تصویر');
                }
                
                // دریافت وضعیت پیشرفت
                $progress = 0;
                $progress_text = 'شروع نشده';
                
                // تلاش برای دریافت وضعیت پیشرفت از Moodle
                $completion_status = $this->moodle_api->get_course_completion_status($moodle_user_id, $course_id);
                
                if ($completion_status && !is_wp_error($completion_status)) {
                    // بررسی ساختارهای مختلف پاسخ Moodle
                    if (isset($completion_status['completionstatus'])) {
                        $status = $completion_status['completionstatus'];
                        
                        if (isset($status['completed']) && $status['completed']) {
                            $progress = 100;
                            $progress_text = 'تکمیل شده';
                        } elseif (isset($status['progress'])) {
                            $progress = intval($status['progress']);
                            if ($progress > 0 && $progress < 100) {
                                $progress_text = $progress . '% تکمیل شده';
                            } else {
                                $progress_text = 'در حال انجام';
                            }
                        } elseif (isset($status['completions']) && is_array($status['completions'])) {
                            $completions = $status['completions'];
                            $total = count($completions);
                            $completed = 0;
                            foreach ($completions as $comp) {
                                // Moodle completion status: 0 = incomplete, 1 = complete
                                if (isset($comp['status']) && $comp['status'] == 1) {
                                    $completed++;
                                }
                            }
                            if ($total > 0) {
                                $progress = round(($completed / $total) * 100);
                                if ($progress == 100) {
                                    $progress_text = 'تکمیل شده';
                                } elseif ($progress > 0) {
                                    $progress_text = $progress . '% تکمیل شده';
                                } else {
                                    $progress_text = 'در حال انجام';
                                }
                            }
                        }
                    } elseif (isset($completion_status['completed'])) {
                        // ساختار ساده‌تر
                        if ($completion_status['completed']) {
                            $progress = 100;
                            $progress_text = 'تکمیل شده';
                        } else {
                            $progress_text = 'در حال انجام';
                        }
                    }
                }
            ?>
                <div class="mws-user-course-card">
                    <a href="<?php echo esc_url($course_url); ?>" target="_blank" class="mws-course-link">
                        <div class="mws-course-thumbnail">
                            <img src="<?php echo esc_url($thumbnail_url); ?>" alt="<?php echo esc_attr($course_name); ?>" />
                        </div>
                        <div class="mws-course-info">
                            <h3 class="mws-course-title"><?php echo esc_html($course_name); ?></h3>
                            <div class="mws-course-progress">
                                <div class="mws-progress-bar">
                                    <div class="mws-progress-fill" style="width: <?php echo esc_attr($progress); ?>%"></div>
                                </div>
                                <span class="mws-progress-text"><?php echo esc_html($progress_text); ?></span>
                            </div>
                        </div>
                    </a>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * شورتکد نمایش تعداد دوره‌های ثبت‌نام شده کاربر
     */
    public function enrolled_count_shortcode($atts) {
        if (!is_user_logged_in()) {
            return '0';
        }
        
        $user_id = get_current_user_id();
        $enrollments = $this->get_user_enrollments($user_id);
        
        // شمارش دوره‌های فعال
        $count = 0;
        if (is_array($enrollments)) {
            foreach ($enrollments as $enrollment) {
                if (!isset($enrollment['status']) || $enrollment['status'] == 'active') {
                    $count++;
                }
            }
        }
        
        return (string) $count;
    }
    
    /**
     * تولید توکن SSO برای کاربر (کمک‌کننده)
     */
    private function generate_sso_token_for_user($user_id, $moodle_user_id) {
        $data = array(
            'user_id' => $user_id,
            'moodle_user_id' => $moodle_user_id,
            'timestamp' => time()
        );
        
        $token = base64_encode(json_encode($data));
        return $token;
    }
    
    /**
     * ثبت لاگ ثبت‌نام
     */
    private function log_enrollment($user_id, $course_id, $action, $status, $message = '') {
        global $wpdb;
        $table_name = $wpdb->prefix . 'mws_sync_log';
        
        $wpdb->insert(
            $table_name,
            array(
                'user_id' => intval($user_id),
                'moodle_user_id' => get_user_meta($user_id, 'mws_moodle_user_id', true),
                'course_id' => intval($course_id),
                'action' => sanitize_text_field($action),
                'status' => sanitize_text_field($status),
                'message' => sanitize_text_field($message),
                'created_at' => current_time('mysql')
            ),
            array('%d', '%d', '%d', '%s', '%s', '%s', '%s')
        );
    }
}

