<?php
/**
 * کلاس همگام‌سازی دوره‌ها بین Moodle و WordPress
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_Course_Sync {
    private $moodle_api;
    
    public function __construct() {
        $this->moodle_api = new MWS_Moodle_API();
    }
    
    /**
     * راه‌اندازی هوک‌ها
     */
    public function init() {
        // شورت‌کد برای نمایش دوره‌ها
        add_shortcode('royal_moodle_courses', array($this, 'display_courses_shortcode'));
        add_shortcode('moodle_courses', array($this, 'display_courses_shortcode')); // برای سازگاری با نسخه قبلی
        
        // شورت‌کد برای نمایش یک دوره
        add_shortcode('royal_moodle_course', array($this, 'display_course_shortcode'));
        add_shortcode('moodle_course', array($this, 'display_course_shortcode')); // برای سازگاری با نسخه قبلی
        
        // همگام‌سازی دوره‌ها از صفحه ادمین
        add_action('wp_ajax_mws_sync_courses', array($this, 'ajax_sync_courses'));
        add_action('wp_ajax_mws_sync_single_course', array($this, 'ajax_sync_single_course'));
        
        // کرون جاب برای همگام‌سازی خودکار
        if (get_option('mws_auto_sync_courses', '1') == '1') {
            add_action('mws_sync_courses_cron', array($this, 'sync_courses_cron'));
            if (!wp_next_scheduled('mws_sync_courses_cron')) {
                wp_schedule_event(time(), 'hourly', 'mws_sync_courses_cron');
            }
        }
        
        // افزودن ستون‌های سفارشی به لیست دوره‌ها
        add_filter('manage_moodle_course_posts_columns', array($this, 'add_course_columns'));
        add_action('manage_moodle_course_posts_custom_column', array($this, 'render_course_columns'), 10, 2);
        add_filter('manage_edit-moodle_course_sortable_columns', array($this, 'sortable_columns'));
        
        // افزودن متا باکس برای دوره‌ها
        add_action('add_meta_boxes', array($this, 'add_course_meta_boxes'));
        add_action('save_post_moodle_course', array($this, 'save_course_meta'));
        
        // همگام‌سازی هنگام ذخیره دوره (وردپرس به مودل)
        add_action('save_post_moodle_course', array($this, 'sync_to_moodle_on_save'), 20);
    }
    
    /**
     * همگام‌سازی دوره‌ها از Moodle
     */
    public function sync_courses($options = array()) {
        if (!$this->moodle_api->is_configured()) {
            return array('success' => false, 'message' => 'Moodle تنظیم نشده است');
        }
        
        // دریافت تنظیمات batch
        $batch_size = isset($options['batch_size']) ? intval($options['batch_size']) : 10;
        $offset = isset($options['offset']) ? intval($options['offset']) : 0;
        $resume = isset($options['resume']) ? $options['resume'] : false;
        
        $courses = $this->moodle_api->get_courses();
        
        if (is_wp_error($courses)) {
            return array('success' => false, 'message' => $courses->get_error_message());
        }
        
        if (!$courses || !is_array($courses)) {
            return array('success' => false, 'message' => 'هیچ دوره‌ای یافت نشد');
        }
        
        // فیلتر کردن دوره پیش‌فرض
        $courses = array_filter($courses, function($course) {
            return isset($course['id']) && $course['id'] != 1;
        });
        
        $total_courses = count($courses);
        $courses = array_values($courses); // بازنشانی کلیدها
        
        // اگر resume فعال است، از آخرین سینک ادامه بده
        if ($resume) {
            $progress = get_option('mws_sync_progress', null);
            if ($progress && isset($progress['processed'])) {
                $offset = $progress['processed'];
            }
        }
        
        // اگر offset مشخص شده، از آنجا شروع کن
        if ($offset > 0 && $offset < $total_courses) {
            $courses = array_slice($courses, $offset);
        }
        
        // محدود کردن به batch_size
        $courses_batch = array_slice($courses, 0, $batch_size);
        
        $synced = 0;
        $updated = 0;
        $skipped = 0;
        $errors = 0;
        
        // ذخیره پیشرفت
        $processed = $offset;
        
        foreach ($courses_batch as $course) {
            try {
            // ایجاد یا به‌روزرسانی دوره
            $result = $this->create_or_update_course($course);
            
                if ($result['success']) {
            if ($result['action'] == 'created') {
                $synced++;
            } elseif ($result['action'] == 'updated') {
                $updated++;
            } else {
                $skipped++;
            }
                } else {
                    $skipped++;
                }
                
                $processed++;
                
                // جلوگیری از timeout
                if (function_exists('set_time_limit')) {
                    @set_time_limit(30);
                }
                
            } catch (Exception $e) {
                $errors++;
                error_log('MWS: خطا در سینک دوره ' . (isset($course['id']) ? $course['id'] : 'unknown') . ': ' . $e->getMessage());
            }
        }
        
        $remaining = $total_courses - $processed;
        $has_more = $remaining > 0;
        
        // ذخیره وضعیت سینک
        update_option('mws_sync_progress', array(
            'processed' => $processed,
            'total' => $total_courses,
            'synced' => $synced,
            'updated' => $updated,
            'skipped' => $skipped,
            'errors' => $errors,
            'has_more' => $has_more
        ));
        
        $message = sprintf(
            'سینک انجام شد. جدید: %d، به‌روزرسانی: %d، رد شده: %d، خطا: %d. پیشرفت: %d/%d',
            $synced, $updated, $skipped, $errors, $processed, $total_courses
        );
        
        if ($has_more) {
            $message .= sprintf(' (%d دوره باقی مانده)', $remaining);
        }
        
        return array(
            'success' => true,
            'message' => $message,
            'synced' => $synced,
            'updated' => $updated,
            'skipped' => $skipped,
            'errors' => $errors,
            'processed' => $processed,
            'total' => $total_courses,
            'has_more' => $has_more,
            'next_offset' => $processed
        );
    }
    
    /**
     * ایجاد یا به‌روزرسانی دوره در WordPress
     */
    private function create_or_update_course($course) {
        // جستجوی دوره موجود
        $args = array(
            'post_type' => 'moodle_course',
            'meta_query' => array(
                array(
                    'key' => 'mws_moodle_course_id',
                    'value' => $course['id'],
                    'compare' => '='
                )
            ),
            'posts_per_page' => 1
        );
        
        $existing = get_posts($args);
        
        $post_data = array(
            'post_title' => sanitize_text_field($course['fullname']),
            'post_content' => wp_kses_post($course['summary']),
            'post_excerpt' => wp_trim_words(wp_kses_post($course['summary']), 30),
            'post_status' => 'publish',
            'post_type' => 'moodle_course'
        );
        
        if (!empty($existing)) {
            $post_data['ID'] = $existing[0]->ID;
            $post_id = wp_update_post($post_data);
            $action = 'updated';
        } else {
            $post_id = wp_insert_post($post_data);
            $action = 'created';
        }
        
        if ($post_id && !is_wp_error($post_id)) {
            // ذخیره متا داده‌ها
            update_post_meta($post_id, 'mws_moodle_course_id', intval($course['id']));
            update_post_meta($post_id, 'mws_course_shortname', sanitize_text_field($course['shortname']));
            update_post_meta($post_id, 'mws_course_category', intval($course['categoryid']));
            update_post_meta($post_id, 'mws_course_startdate', intval($course['startdate']));
            update_post_meta($post_id, 'mws_course_enddate', intval($course['enddate']));
            update_post_meta($post_id, 'mws_course_visible', intval($course['visible']));
            update_post_meta($post_id, 'mws_synced_at', current_time('mysql'));
            
            // تنظیم دسته‌بندی دوره
            $this->set_course_category($post_id, intval($course['categoryid']));
            
            return array('success' => true, 'action' => $action, 'post_id' => $post_id);
        }
        
        return array('success' => false, 'action' => 'failed');
    }
    
    /**
     * نمایش دوره‌ها با شورت‌کد
     */
    public function display_courses_shortcode($atts) {
        $atts = shortcode_atts(array(
            'category' => '',
            'limit' => 10,
            'columns' => 3,
            'show_enroll' => 'yes'
        ), $atts);
        
        $args = array(
            'post_type' => 'moodle_course',
            'posts_per_page' => intval($atts['limit']),
            'post_status' => 'publish',
            'orderby' => 'date',
            'order' => 'DESC'
        );
        
        // فیلتر بر اساس دسته‌بندی
        if (!empty($atts['category'])) {
            $args['meta_query'] = array(
                array(
                    'key' => 'mws_course_category',
                    'value' => intval($atts['category']),
                    'compare' => '='
                )
            );
        }
        
        $courses = get_posts($args);
        
        if (empty($courses)) {
            return '<p>هیچ دوره‌ای یافت نشد.</p>';
        }
        
        ob_start();
        ?>
        <div class="mws-courses-list columns-<?php echo esc_attr($atts['columns']); ?>">
            <?php foreach ($courses as $course) : 
                $moodle_id = get_post_meta($course->ID, 'mws_moodle_course_id', true);
                $startdate = get_post_meta($course->ID, 'mws_course_startdate', true);
                $is_enrolled = false;
                
                if (is_user_logged_in()) {
                    $user_id = get_current_user_id();
                    $enrollments = get_user_meta($user_id, 'mws_enrollments', true);
                    if (is_array($enrollments)) {
                        foreach ($enrollments as $enrollment) {
                            if ($enrollment['course_id'] == $moodle_id) {
                                $is_enrolled = true;
                                break;
                            }
                        }
                    }
                }
            ?>
                <div class="mws-course-item">
                    <div class="mws-course-content">
                        <h3 class="mws-course-title">
                            <a href="<?php echo esc_url(get_permalink($course->ID)); ?>">
                                <?php echo esc_html($course->post_title); ?>
                            </a>
                        </h3>
                        <div class="mws-course-excerpt">
                            <?php echo wp_kses_post($course->post_excerpt ?: wp_trim_words($course->post_content, 20)); ?>
                        </div>
                        <?php if ($startdate) : ?>
                            <div class="mws-course-date">
                                تاریخ شروع: <?php echo date_i18n('j F Y', $startdate); ?>
                            </div>
                        <?php endif; ?>
                        <?php if ($atts['show_enroll'] == 'yes') : ?>
                            <div class="mws-course-actions">
                                <?php if (is_user_logged_in()) : ?>
                                    <?php if ($is_enrolled) : ?>
                                        <span class="mws-enrolled-badge">شما در این دوره ثبت‌نام کرده‌اید</span>
                                    <?php else : ?>
                                        <a href="<?php echo esc_url(add_query_arg('enroll', $moodle_id)); ?>" class="mws-btn-enroll">
                                            ثبت‌نام در دوره
                                        </a>
                                    <?php endif; ?>
                                <?php else : ?>
                                    <a href="<?php echo esc_url(wp_login_url(get_permalink($course->ID))); ?>" class="mws-btn-login">
                                        برای ثبت‌نام وارد شوید
                                    </a>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * نمایش یک دوره با شورت‌کد
     */
    public function display_course_shortcode($atts) {
        $atts = shortcode_atts(array(
            'id' => '',
            'moodle_id' => ''
        ), $atts);
        
        if (empty($atts['id']) && empty($atts['moodle_id'])) {
            return '<p>شناسه دوره مشخص نشده است.</p>';
        }
        
        $args = array(
            'post_type' => 'moodle_course',
            'posts_per_page' => 1,
            'post_status' => 'publish'
        );
        
        if (!empty($atts['id'])) {
            $args['p'] = intval($atts['id']);
        } else {
            $args['meta_query'] = array(
                array(
                    'key' => 'mws_moodle_course_id',
                    'value' => intval($atts['moodle_id']),
                    'compare' => '='
                )
            );
        }
        
        $courses = get_posts($args);
        
        if (empty($courses)) {
            return '<p>دوره یافت نشد.</p>';
        }
        
        $course = $courses[0];
        $moodle_id = get_post_meta($course->ID, 'mws_moodle_course_id', true);
        
        ob_start();
        ?>
        <div class="mws-single-course">
            <h2><?php echo esc_html($course->post_title); ?></h2>
            <div class="mws-course-content">
                <?php echo wp_kses_post($course->post_content); ?>
            </div>
            <?php echo do_shortcode('[moodle_enroll_button course_id="' . esc_attr($moodle_id) . '"]'); ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * همگام‌سازی خودکار با کرون
     */
    public function sync_courses_cron() {
        $this->sync_courses();
    }
    
    /**
     * همگام‌سازی دوره‌ها (AJAX)
     */
    public function ajax_sync_courses() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $batch_size = isset($_POST['batch_size']) ? intval($_POST['batch_size']) : 10;
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        $resume = isset($_POST['resume']) && $_POST['resume'] == '1';
        
        $options = array(
            'batch_size' => $batch_size,
            'offset' => $offset,
            'resume' => $resume
        );
        
        $result = $this->sync_courses($options);
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    /**
     * افزودن متا باکس‌های دوره
     */
    public function add_course_meta_boxes() {
        add_meta_box(
            'mws_course_info',
            'اطلاعات دوره Moodle',
            array($this, 'course_info_meta_box'),
            'moodle_course',
            'side',
            'high'
        );
        
        add_meta_box(
            'mws_course_settings',
            'تنظیمات دوره',
            array($this, 'course_settings_meta_box'),
            'moodle_course',
            'normal',
            'high'
        );
        
    }
    
    /**
     * نمایش متا باکس اطلاعات دوره
     */
    public function course_info_meta_box($post) {
        $moodle_id = get_post_meta($post->ID, 'mws_moodle_course_id', true);
        $shortname = get_post_meta($post->ID, 'mws_course_shortname', true);
        $category = get_post_meta($post->ID, 'mws_course_category', true);
        $synced_at = get_post_meta($post->ID, 'mws_synced_at', true);
        
        wp_nonce_field('mws_course_meta', 'mws_course_meta_nonce');
        ?>
        <table class="form-table">
            <tr>
                <th><label>شناسه Moodle:</label></th>
                <td><strong><?php echo esc_html($moodle_id ?: 'هنوز سینک نشده'); ?></strong></td>
            </tr>
            <tr>
                <th><label>نام کوتاه:</label></th>
                <td><?php echo esc_html($shortname ?: '-'); ?></td>
            </tr>
            <tr>
                <th><label>دسته‌بندی Moodle:</label></th>
                <td>
                    <?php
                    $categories = get_terms(array(
                        'taxonomy' => 'moodle_category',
                        'hide_empty' => false
                    ));
                    if (!empty($categories)) :
                        $selected_category = $category;
                    ?>
                        <select name="mws_course_category" id="mws_course_category" class="regular-text">
                            <option value="">-- انتخاب دسته‌بندی --</option>
                            <?php foreach ($categories as $cat) :
                                $moodle_cat_id = get_term_meta($cat->term_id, 'mws_moodle_category_id', true);
                            ?>
                                <option value="<?php echo esc_attr($moodle_cat_id); ?>" <?php selected($selected_category, $moodle_cat_id); ?>>
                                    <?php echo esc_html($cat->name . ' (ID: ' . $moodle_cat_id . ')'); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    <?php else : ?>
                        <p style="color: #999;">ابتدا دسته‌بندی‌ها را سینک کنید</p>
                    <?php endif; ?>
                </td>
            </tr>
            <tr>
                <th><label>آخرین همگام‌سازی:</label></th>
                <td><?php echo esc_html($synced_at ?: '-'); ?></td>
            </tr>
        </table>
        <p>
            <label>
                <input type="checkbox" name="mws_sync_to_moodle" value="1" />
                همگام‌سازی با Moodle هنگام ذخیره
            </label>
        </p>
        <?php if ($moodle_id) : ?>
            <p>
                <a href="<?php echo esc_url(add_query_arg('enroll', $moodle_id)); ?>" class="button">
                    تست ثبت‌نام
                </a>
            </p>
        <?php endif; ?>
        <?php
    }
    
    /**
     * نمایش متا باکس تنظیمات دوره
     */
    public function course_settings_meta_box($post) {
        $course_type = get_post_meta($post->ID, 'mws_course_type', true);
        $course_price = get_post_meta($post->ID, 'mws_course_price', true);
        $course_url = get_post_meta($post->ID, 'mws_course_url', true);
        $course_duration = get_post_meta($post->ID, 'mws_course_duration_days', true);
        
        wp_nonce_field('mws_course_meta', 'mws_course_meta_nonce');
        ?>
        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="mws_course_type">نوع دوره</label>
                </th>
                <td>
                    <select name="mws_course_type" id="mws_course_type" class="regular-text">
                        <option value="free" <?php selected($course_type, 'free'); ?>>رایگان</option>
                        <option value="paid" <?php selected($course_type, 'paid'); ?>>پولی</option>
                        <option value="closed" <?php selected($course_type, 'closed'); ?>>بسته شده</option>
                    </select>
                    <p class="description">نوع دسترسی به دوره را انتخاب کنید</p>
                </td>
            </tr>
            <tr id="mws_price_row" style="<?php echo $course_type != 'paid' ? 'display: none;' : ''; ?>">
                <th scope="row">
                    <label for="mws_course_price">قیمت (تومان)</label>
                </th>
                <td>
                    <input type="number" 
                           name="mws_course_price" 
                           id="mws_course_price" 
                           value="<?php echo esc_attr($course_price); ?>" 
                           class="regular-text" 
                           min="0" 
                           step="1000" />
                    <p class="description">قیمت دوره را به تومان وارد کنید</p>
                </td>
            </tr>
            <tr id="mws_url_row" style="<?php echo $course_type != 'closed' ? 'display: none;' : ''; ?>">
                <th scope="row">
                    <label for="mws_course_url">آدرس URL</label>
                </th>
                <td>
                    <input type="url" 
                           name="mws_course_url" 
                           id="mws_course_url" 
                           value="<?php echo esc_url($course_url); ?>" 
                           class="regular-text" 
                           placeholder="https://example.com/course" />
                    <p class="description">آدرس URL دوره خارجی را وارد کنید. کاربران با کلیک روی دوره به این آدرس هدایت می‌شوند.</p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="mws_course_duration_days">مدت دسترسی (روز)</label>
                </th>
                <td>
                    <input type="number" 
                           name="mws_course_duration_days" 
                           id="mws_course_duration_days" 
                           value="<?php echo esc_attr($course_duration); ?>" 
                           class="small-text" 
                           min="0" 
                           step="1" 
                           placeholder="0" />
                    <p class="description">
                        تعداد روزهای دسترسی پس از خرید را وارد کنید.<br>
                        مثال: <strong>30</strong> = یک ماه، <strong>365</strong> = یک سال.<br>
                        <strong>0</strong> یا خالی = دسترسی نامحدود.
                    </p>
                </td>
            </tr>
        </table>
        <script>
        jQuery(document).ready(function($) {
            $('#mws_course_type').on('change', function() {
                var courseType = $(this).val();
                
                if (courseType == 'paid') {
                    $('#mws_price_row').show();
                    $('#mws_url_row').hide();
                } else if (courseType == 'closed') {
                    $('#mws_price_row').hide();
                    $('#mws_url_row').show();
                } else {
                    $('#mws_price_row').hide();
                    $('#mws_url_row').hide();
                }
            });
        });
        </script>
        <?php
    }
    
    /**
     * افزودن ستون‌های سفارشی به لیست دوره‌ها
     */
    public function add_course_columns($columns) {
        $new_columns = array();
        $new_columns['cb'] = $columns['cb'];
        $new_columns['mws_featured_image'] = 'تصویر شاخص';
        $new_columns['title'] = 'عنوان دوره';
        $new_columns['mws_moodle_id'] = 'شناسه Moodle';
        $new_columns['mws_sync'] = 'سینک دستی';
        $new_columns['mws_course_type'] = 'نوع دوره';
        $new_columns['mws_course_price'] = 'قیمت';
        $new_columns['date'] = 'تاریخ انتشار';
        
        return $new_columns;
    }
    
    /**
     * نمایش محتوای ستون‌های سفارشی
     */
    public function render_course_columns($column, $post_id) {
        switch ($column) {
            case 'mws_featured_image':
                if (has_post_thumbnail($post_id)) {
                    $thumbnail = get_the_post_thumbnail($post_id, array(60, 60), array('style' => 'max-width: 60px; height: auto;'));
                    echo $thumbnail;
                } else {
                    echo '<span style="display: inline-block; width: 60px; height: 60px; background: #f0f0f0; border: 1px solid #ddd; text-align: center; line-height: 60px; color: #999; font-size: 11px;" title="بدون تصویر شاخص">—</span>';
                }
                break;
                
            case 'mws_moodle_id':
                $moodle_id = get_post_meta($post_id, 'mws_moodle_course_id', true);
                if ($moodle_id) {
                    echo '<strong>' . esc_html($moodle_id) . '</strong>';
                } else {
                    echo '<span style="color: #999;">هنوز سینک نشده</span>';
                }
                break;
                
            case 'mws_sync':
                $moodle_id = get_post_meta($post_id, 'mws_moodle_course_id', true);
                echo '<button type="button" class="button button-small mws-sync-single-course" data-post-id="' . esc_attr($post_id) . '">';
                echo $moodle_id ? 'به‌روزرسانی' : 'سینک به Moodle';
                echo '</button>';
                echo '<span class="mws-sync-status-' . esc_attr($post_id) . '" style="margin-right: 5px;"></span>';
                break;
                
            case 'mws_course_type':
                $course_type = get_post_meta($post_id, 'mws_course_type', true);
                $type_labels = array(
                    'free' => 'رایگان',
                    'paid' => 'پولی',
                    'closed' => 'بسته شده'
                );
                $label = isset($type_labels[$course_type]) ? $type_labels[$course_type] : 'نامشخص';
                $colors = array(
                    'free' => '#00a32a',
                    'paid' => '#2271b1',
                    'closed' => '#d63638'
                );
                $color = isset($colors[$course_type]) ? $colors[$course_type] : '#666';
                echo '<span style="color: ' . esc_attr($color) . '; font-weight: bold;">' . esc_html($label) . '</span>';
                break;
                
            case 'mws_course_price':
                $course_type = get_post_meta($post_id, 'mws_course_type', true);
                if ($course_type == 'paid') {
                    $price = get_post_meta($post_id, 'mws_course_price', true);
                    if ($price) {
                        echo number_format($price, 0) . ' تومان';
                    } else {
                        echo '<span style="color: #999;">قیمت تنظیم نشده</span>';
                    }
                } else {
                    echo '-';
                }
                break;
        }
    }
    
    /**
     * ستون‌های قابل مرتب‌سازی
     */
    public function sortable_columns($columns) {
        $columns['mws_moodle_id'] = 'mws_moodle_id';
        $columns['mws_course_type'] = 'mws_course_type';
        return $columns;
    }
    
    /**
     * ذخیره متا داده‌های دوره
     */
    public function save_course_meta($post_id) {
        if (!isset($_POST['mws_course_meta_nonce']) || 
            !wp_verify_nonce($_POST['mws_course_meta_nonce'], 'mws_course_meta')) {
            return;
        }
        
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // ذخیره نوع دوره
        if (isset($_POST['mws_course_type'])) {
            update_post_meta($post_id, 'mws_course_type', sanitize_text_field($_POST['mws_course_type']));
        }
        
        // ذخیره قیمت
        if (isset($_POST['mws_course_price'])) {
            $price = floatval($_POST['mws_course_price']);
            update_post_meta($post_id, 'mws_course_price', $price);
        }
        
        // ذخیره URL (برای حالت closed)
        if (isset($_POST['mws_course_url'])) {
            $url = esc_url_raw($_POST['mws_course_url']);
            if (!empty($url)) {
                update_post_meta($post_id, 'mws_course_url', $url);
            } else {
                delete_post_meta($post_id, 'mws_course_url');
            }
        }
        
        // ذخیره مدت دسترسی (روز)
        if (isset($_POST['mws_course_duration_days'])) {
            $duration = max(0, intval($_POST['mws_course_duration_days']));
            update_post_meta($post_id, 'mws_course_duration_days', $duration);
        }
        
        // ذخیره دسته‌بندی Moodle
        if (isset($_POST['mws_course_category'])) {
            update_post_meta($post_id, 'mws_course_category', intval($_POST['mws_course_category']));
        }
    }
    
    /**
     * همگام‌سازی دوره به Moodle هنگام ذخیره
     */
    public function sync_to_moodle_on_save($post_id) {
        // جلوگیری از اجرای خودکار در برخی حالات
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // بررسی اینکه آیا باید به Moodle سینک شود
        if (!isset($_POST['mws_sync_to_moodle']) || $_POST['mws_sync_to_moodle'] != '1') {
            return;
        }
        
        $post = get_post($post_id);
        if ($post->post_type != 'moodle_course' || $post->post_status != 'publish') {
            return;
        }
        
        // همگام‌سازی به Moodle
        $this->create_course_in_moodle($post_id);
    }
    
    /**
     * ایجاد یا به‌روزرسانی دوره در Moodle
     */
    public function create_course_in_moodle($post_id) {
        if (!$this->moodle_api->is_configured()) {
            return false;
        }
        
        $post = get_post($post_id);
        $moodle_id = get_post_meta($post_id, 'mws_moodle_course_id', true);
        $category_id = get_post_meta($post_id, 'mws_course_category', true);
        
        if (!$category_id) {
            // استفاده از دسته‌بندی پیش‌فرض
            $category_id = 1;
        }
        
        $shortname = get_post_meta($post_id, 'mws_course_shortname', true);
        if (empty($shortname)) {
            $shortname = sanitize_title($post->post_title);
        }
        
        $course_data = array(
            'fullname' => $post->post_title,
            'shortname' => $shortname,
            'summary' => $post->post_content,
            'categoryid' => intval($category_id),
            'visible' => 1
        );
        
        if ($moodle_id) {
            // به‌روزرسانی دوره موجود
            $course_data['id'] = intval($moodle_id);
            $result = $this->moodle_api->update_course($course_data);
        } else {
            // ایجاد دوره جدید
            $result = $this->moodle_api->create_course($course_data);
        }
        
        if (!is_wp_error($result)) {
            if (!$moodle_id && isset($result['id'])) {
                update_post_meta($post_id, 'mws_moodle_course_id', intval($result['id']));
            }
            update_post_meta($post_id, 'mws_synced_at', current_time('mysql'));
            return true;
        }
        
        return false;
    }
    
    /**
     * همگام‌سازی یک دوره از Moodle (AJAX)
     */
    public function ajax_sync_single_course() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        
        if (!$post_id) {
            wp_send_json_error(array('message' => 'شناسه دوره نامعتبر است'));
        }
        
        // دریافت moodle_id از پست
            $moodle_id = get_post_meta($post_id, 'mws_moodle_course_id', true);
        
        if (!$moodle_id) {
            wp_send_json_error(array('message' => 'این دوره هنوز با Moodle سینک نشده است. لطفاً ابتدا دوره را از Moodle سینک کنید.'));
        }
        
        // دریافت دوره از Moodle
        $course = $this->moodle_api->get_course($moodle_id);
        
        if (!$course || is_wp_error($course)) {
            wp_send_json_error(array('message' => 'خطا در دریافت دوره از Moodle: ' . (is_wp_error($course) ? $course->get_error_message() : 'دوره یافت نشد')));
        }
        
        // به‌روزرسانی دوره در WordPress
        $result = $this->create_or_update_course($course);
        
        if ($result['success']) {
            wp_send_json_success(array(
                'message' => 'همگام‌سازی با موفقیت انجام شد',
                'action' => $result['action'],
                'moodle_id' => $moodle_id,
                'post_id' => $post_id
            ));
        } else {
            wp_send_json_error(array('message' => 'خطا در به‌روزرسانی دوره در WordPress'));
        }
    }
    
    /**
     * تنظیم دسته‌بندی دوره بر اساس categoryid از Moodle
     */
    private function set_course_category($post_id, $moodle_category_id) {
        if (!$moodle_category_id || $moodle_category_id <= 0) {
            return false;
        }
        
        // جستجوی term مربوط به این categoryid در taxonomy
        $terms = get_terms(array(
            'taxonomy' => 'moodle_category',
            'hide_empty' => false,
            'meta_query' => array(
                array(
                    'key' => 'mws_moodle_category_id',
                    'value' => $moodle_category_id,
                    'compare' => '='
                )
            )
        ));
        
        if (!empty($terms) && !is_wp_error($terms)) {
            // پیدا کردن term_id
            $term_id = $terms[0]->term_id;
            
            // تنظیم دسته‌بندی برای پست
            $result = wp_set_post_terms($post_id, array($term_id), 'moodle_category', false);
            
            if (!is_wp_error($result)) {
                return true;
            } else {
                error_log('MWS: خطا در تنظیم دسته‌بندی دوره: ' . $result->get_error_message());
            }
        } else {
            // اگر دسته‌بندی پیدا نشد، می‌توانید یک دسته‌بندی پیش‌فرض ایجاد کنید
            // یا فقط لاگ کنید که دسته‌بندی یافت نشد
            error_log('MWS: دسته‌بندی Moodle با ID ' . $moodle_category_id . ' در WordPress یافت نشد. لطفاً ابتدا دسته‌بندی‌ها را سینک کنید.');
        }
        
        return false;
    }
}

