<?php
/**
 * مدیریت دانشجویان دوره و نمایش دوره‌ها در پروفایل کاربر
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_Course_Students {

    private $moodle_api;

    public function __construct() {
        if (class_exists('MWS_Moodle_API')) {
            $this->moodle_api = new MWS_Moodle_API();
        }
    }

    public function init() {
        add_action('add_meta_boxes', array($this, 'add_students_meta_box'));

        add_action('show_user_profile', array($this, 'user_courses_profile_section'));
        add_action('edit_user_profile', array($this, 'user_courses_profile_section'));

        add_action('wp_ajax_mws_get_course_students',          array($this, 'ajax_get_course_students'));
        add_action('wp_ajax_mws_get_student_grade',            array($this, 'ajax_get_student_grade'));
        add_action('wp_ajax_mws_course_enroll_user',           array($this, 'ajax_enroll_user'));
        add_action('wp_ajax_mws_course_unenroll_user',         array($this, 'ajax_unenroll_user'));
        add_action('wp_ajax_mws_course_update_expiry',         array($this, 'ajax_update_expiry'));
        add_action('wp_ajax_mws_sync_user_enrollments_to_meta', array($this, 'ajax_sync_user_enrollments'));
    }

    /* ------------------------------------------------------------------ */
    /*  تبدیل تاریخ به شمسی                                               */
    /* ------------------------------------------------------------------ */

    private function greg_to_jal($g_y, $g_m, $g_d) {
        $g_y_m  = array(0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334);
        $jy     = ($g_y <= 1600) ? 0 : 979;
        $g_y    = ($g_y <= 1600) ? ($g_y - 621) : ($g_y - 1600);
        $g_y2   = ($g_m > 2) ? ($g_y + 1) : $g_y;
        $gy_sum = 365 * $g_y
                  + intdiv($g_y2 + 3,   4)
                  - intdiv($g_y2 + 99,  100)
                  + intdiv($g_y2 + 399, 400)
                  - 80 + $g_d + $g_y_m[$g_m - 1];

        $jy += 33 * intdiv($gy_sum, 12053);
        $gy_sum %= 12053;
        $jy     += 4 * intdiv($gy_sum, 1461);
        $gy_sum %= 1461;

        if ($gy_sum > 365) {
            $jy    += intdiv($gy_sum - 1, 365);
            $gy_sum = ($gy_sum - 1) % 365;
        }

        $j_y_m = array(0, 31, 62, 93, 124, 155, 186, 216, 246, 276, 306, 336);
        $jm = 0; $jd = 0;
        for ($i = 11; $i >= 0; $i--) {
            if ($gy_sum >= $j_y_m[$i]) {
                $jm = $i + 1;
                $jd = $gy_sum - $j_y_m[$i] + 1;
                break;
            }
        }
        return array($jy, $jm, $jd);
    }

    private function to_jalali($date_string) {
        if (empty($date_string)) {
            return '';
        }
        $ts = is_numeric($date_string) ? intval($date_string) : strtotime($date_string);
        if (!$ts || $ts <= 0) {
            return '';
        }
        $d = getdate($ts);
        list($jy, $jm, $jd) = $this->greg_to_jal($d['year'], $d['mon'], $d['mday']);
        return sprintf('%04d/%02d/%02d', $jy, $jm, $jd);
    }

    /* ------------------------------------------------------------------ */
    /*  متاباکس دانشجویان                                                  */
    /* ------------------------------------------------------------------ */

    public function add_students_meta_box() {
        add_meta_box(
            'mws_course_students',
            'دانشجویان دوره',
            array($this, 'students_meta_box_content'),
            'moodle_course',
            'normal',
            'low'
        );
    }

    public function students_meta_box_content($post) {
        $moodle_course_id = get_post_meta($post->ID, 'mws_moodle_course_id', true);
        $nonce            = wp_create_nonce('mws_ajax_nonce');

        if (!$moodle_course_id) {
            echo '<p style="color:#999;padding:10px;">ابتدا دوره را با Moodle همگام‌سازی کنید تا شناسه Moodle تنظیم شود.</p>';
            return;
        }
        ?>
        <div class="mws-students-box"
             data-course-id="<?php echo esc_attr($moodle_course_id); ?>"
             data-nonce="<?php echo esc_attr($nonce); ?>">

            <div class="mws-enroll-form-row"
                 style="display:flex;align-items:center;gap:12px;margin-bottom:16px;padding-bottom:16px;border-bottom:1px solid #e0e0e0;flex-wrap:wrap;">
                <label style="font-weight:600;white-space:nowrap;">اتصال کاربر:</label>
                <select id="mws-enroll-user-select"
                        style="width:320px;"
                        data-placeholder="جستجو بر اساس نام یا ایمیل...">
                    <option value=""></option>
                </select>
                <button type="button" id="mws-do-enroll-btn" class="button button-primary">
                    اتصال به دوره
                </button>
                <span id="mws-enroll-result" style="font-size:13px;"></span>
            </div>

            <div style="display:flex;align-items:center;gap:12px;margin-bottom:12px;">
                <input type="text"
                       id="mws-student-search"
                       placeholder="جستجو در لیست (نام، ایمیل)..."
                       style="width:280px;"
                       class="regular-text" />
                <button type="button" id="mws-load-grades-btn" class="button button-small">
                    دریافت همه نمرات
                </button>
                <span id="mws-student-count" style="color:#666;font-size:13px;"></span>
            </div>

            <div id="mws-students-table-wrap">
                <p style="color:#666;">
                    <span class="spinner is-active" style="float:none;margin-left:6px;"></span>
                    در حال بارگذاری لیست دانشجویان...
                </p>
            </div>
        </div>

        <style>
        .mws-students-tbl { width:100%; border-collapse:collapse; font-size:13px; }
        .mws-students-tbl th { background:#f0f0f1; padding:8px 10px; text-align:right; border-bottom:2px solid #c3c4c7; }
        .mws-students-tbl td { padding:7px 10px; border-bottom:1px solid #ebebeb; vertical-align:middle; }
        .mws-students-tbl tr:hover td { background:#f6f7f7; }
        .mws-badge-active   { color:#00a32a; font-weight:600; }
        .mws-badge-suspended { color:#d63638; font-weight:600; }
        .mws-ops-cell button { display:block; width:100%; margin-bottom:3px; font-size:11px; }
        </style>

        <script>
        jQuery(document).ready(function($) {
            var courseId   = $('.mws-students-box').data('course-id');
            var nonce      = $('.mws-students-box').data('nonce');
            var allStudents = [];

            /* ── Select2 جستجوی کاربر ── */
            if ($.fn.select2) {
                $('#mws-enroll-user-select').select2({
                    placeholder: 'جستجو بر اساس نام یا ایمیل...',
                    allowClear: true,
                    minimumInputLength: 2,
                    ajax: {
                        url: ajaxurl, dataType: 'json', delay: 250,
                        data: function(p) {
                            return { q: p.term || '', page: p.page || 1,
                                     action: 'mws_search_users', nonce: nonce };
                        },
                        processResults: function(d) {
                            return { results: d.results || [],
                                     pagination: { more: d.pagination && d.pagination.more } };
                        },
                        cache: true
                    },
                    language: { noResults: function(){ return 'نتیجه‌ای یافت نشد'; },
                                searching: function(){ return 'در حال جستجو...'; } }
                });
            }

            /* ── بارگذاری دانشجویان ── */
            loadStudents();

            function loadStudents() {
                $('#mws-students-table-wrap').html(
                    '<p style="color:#666;"><span class="spinner is-active" style="float:none;margin-left:6px;"></span> در حال بارگذاری...</p>'
                );
                $.ajax({
                    url: ajaxurl, type: 'POST',
                    data: { action: 'mws_get_course_students', course_id: courseId, nonce: nonce },
                    success: function(res) {
                        if (res.success) {
                            allStudents = res.data.students || [];
                            renderTable(allStudents);
                            $('#mws-student-count').text('(' + allStudents.length + ' نفر)');
                        } else {
                            $('#mws-students-table-wrap').html(
                                '<p style="color:#d63638;">خطا: ' + (res.data.message || 'خطا در بارگذاری') + '</p>'
                            );
                        }
                    },
                    error: function() {
                        $('#mws-students-table-wrap').html('<p style="color:#d63638;">خطا در ارتباط با سرور</p>');
                    }
                });
            }

            function renderTable(students) {
                if (!students.length) {
                    $('#mws-students-table-wrap').html(
                        '<p style="color:#999;padding:20px;text-align:center;">هیچ دانشجویی یافت نشد.</p>'
                    );
                    return;
                }
                var html = '<table class="mws-students-tbl"><thead><tr>'
                    + '<th>نام و نام‌خانوادگی</th>'
                    + '<th>ایمیل</th>'
                    + '<th>تاریخ ثبت‌نام</th>'
                    + '<th>تاریخ پایان</th>'
                    + '<th>وضعیت</th>'
                    + '<th>نمره</th>'
                    + '<th style="width:110px;">عملیات</th>'
                    + '</tr></thead><tbody>';

                students.forEach(function(s) {
                    var statusHtml = s.status === 'suspended'
                        ? '<span class="mws-badge-suspended">تعلیق</span>'
                        : '<span class="mws-badge-active">فعال</span>';

                    var expiryHtml;
                    if (s.expires_at) {
                        var color = s.expired ? '#d63638' : '#00a32a';
                        expiryHtml = '<span style="color:' + color + ';">'
                            + s.expires_at + (s.expired ? ' ⚠' : '') + '</span>';
                    } else {
                        expiryHtml = '<span style="color:#999;">نامحدود</span>';
                    }

                    html += '<tr data-uid="' + s.moodle_user_id + '">'
                        + '<td><strong>' + $('<span>').text(s.fullname).html() + '</strong></td>'
                        + '<td>' + $('<span>').text(s.email).html() + '</td>'
                        + '<td>' + (s.enrolled_at || '<span style="color:#999;">نامشخص</span>') + '</td>'
                        + '<td class="mws-exp-' + s.moodle_user_id + '">' + expiryHtml + '</td>'
                        + '<td>' + statusHtml + '</td>'
                        + '<td class="mws-grade-' + s.moodle_user_id + '">'
                        +   '<button type="button" class="button button-small mws-grade-btn" '
                        +   'data-mid="' + s.moodle_user_id + '">نمره</button>'
                        + '</td>'
                        + '<td class="mws-ops-cell">'
                        +   '<button type="button" class="button button-small mws-expiry-btn" '
                        +   'data-mid="' + s.moodle_user_id + '" '
                        +   'data-wid="' + s.wp_user_id + '" '
                        +   'data-exp="' + (s.expires_at_raw || '') + '">تغییر پایان</button>'
                        +   '<button type="button" class="button button-small mws-unenroll-btn" '
                        +   'data-mid="' + s.moodle_user_id + '" '
                        +   'data-wid="' + s.wp_user_id + '" '
                        +   'style="color:#d63638;">قطع اتصال</button>'
                        + '</td>'
                        + '</tr>';
                });
                html += '</tbody></table>';
                $('#mws-students-table-wrap').html(html);
            }

            /* ── سرچ ── */
            $('#mws-student-search').on('input', function() {
                var q = $(this).val().toLowerCase();
                if (!q) { renderTable(allStudents); return; }
                renderTable(allStudents.filter(function(s) {
                    return s.fullname.toLowerCase().indexOf(q) >= 0
                        || s.email.toLowerCase().indexOf(q) >= 0;
                }));
            });

            /* ── دریافت نمره یک نفر ── */
            $(document).on('click', '.mws-grade-btn', function() {
                var $btn = $(this), mid = $btn.data('mid');
                $btn.prop('disabled', true).text('...');
                $.ajax({
                    url: ajaxurl, type: 'POST',
                    data: { action: 'mws_get_student_grade', course_id: courseId,
                            moodle_user_id: mid, nonce: nonce },
                    success: function(res) {
                        if (res.success) {
                            $('.mws-grade-' + mid).html(
                                res.data.grade
                                    ? '<strong>' + res.data.grade + '</strong>'
                                    : '<span style="color:#999;">—</span>'
                            );
                        } else { $btn.prop('disabled', false).text('نمره'); }
                    },
                    error: function() { $btn.prop('disabled', false).text('نمره'); }
                });
            });

            /* ── دریافت همه نمرات ── */
            $('#mws-load-grades-btn').on('click', function() {
                $('.mws-grade-btn:not([disabled])').each(function() { $(this).trigger('click'); });
            });

            /* ── قطع اتصال ── */
            $(document).on('click', '.mws-unenroll-btn', function() {
                if (!confirm('آیا مطمئن هستید که می‌خواهید این دانشجو را از دوره حذف کنید؟')) return;
                var $btn = $(this), mid = $btn.data('mid'), wid = $btn.data('wid');
                $btn.prop('disabled', true).text('...');
                $.ajax({
                    url: ajaxurl, type: 'POST',
                    data: { action: 'mws_course_unenroll_user', course_id: courseId,
                            moodle_user_id: mid, wp_user_id: wid, nonce: nonce },
                    success: function(res) {
                        if (res.success) {
                            $('tr[data-uid="' + mid + '"]').fadeOut();
                            allStudents = allStudents.filter(function(s) { return s.moodle_user_id != mid; });
                            $('#mws-student-count').text('(' + allStudents.length + ' نفر)');
                        } else {
                            alert('خطا: ' + (res.data.message || 'خطا در قطع اتصال'));
                            $btn.prop('disabled', false).text('قطع اتصال');
                        }
                    },
                    error: function() { $btn.prop('disabled', false).text('قطع اتصال'); }
                });
            });

            /* ── تغییر تاریخ پایان ── */
            $(document).on('click', '.mws-expiry-btn', function() {
                var $btn = $(this), mid = $btn.data('mid'), wid = $btn.data('wid'), exp = $btn.data('exp');
                var info = exp ? 'تاریخ پایان فعلی:\n' + exp : 'تاریخ پایانی تنظیم نشده (نامحدود)';
                var days = prompt(info + '\n\nتعداد روز وارد کنید:\n(مثبت = اضافه | منفی = کم)', '30');
                if (!days) return;
                days = parseInt(days);
                if (isNaN(days) || days === 0) { alert('مقدار نامعتبر'); return; }
                $btn.prop('disabled', true).text('...');
                $.ajax({
                    url: ajaxurl, type: 'POST',
                    data: { action: 'mws_course_update_expiry', course_id: courseId,
                            moodle_user_id: mid, wp_user_id: wid, days: days, nonce: nonce },
                    success: function(res) {
                        if (res.success) {
                            $btn.data('exp', res.data.expires_at_raw || '');
                            var html = res.data.expires_at
                                ? '<span style="color:#00a32a;">' + res.data.expires_at + '</span>'
                                : '<span style="color:#999;">نامحدود</span>';
                            $('.mws-exp-' + mid).html(html);
                        } else { alert('خطا: ' + (res.data.message || 'خطا')); }
                        $btn.prop('disabled', false).text('تغییر پایان');
                    },
                    error: function() { $btn.prop('disabled', false).text('تغییر پایان'); }
                });
            });

            /* ── اتصال کاربر جدید ── */
            $('#mws-do-enroll-btn').on('click', function() {
                var userId = $('#mws-enroll-user-select').val();
                if (!userId) { alert('لطفاً یک کاربر انتخاب کنید'); return; }
                var $btn = $(this);
                $btn.prop('disabled', true).text('در حال اتصال...');
                $.ajax({
                    url: ajaxurl, type: 'POST',
                    data: { action: 'mws_course_enroll_user', course_id: courseId,
                            user_id: userId, nonce: nonce },
                    success: function(res) {
                        if (res.success) {
                            $('#mws-enroll-result').html('<span style="color:#00a32a;">✓ ' + res.data.message + '</span>');
                            loadStudents();
                        } else {
                            $('#mws-enroll-result').html('<span style="color:#d63638;">✗ ' + (res.data.message || 'خطا') + '</span>');
                        }
                        $btn.prop('disabled', false).text('اتصال به دوره');
                        setTimeout(function() { $('#mws-enroll-result').html(''); }, 4000);
                    },
                    error: function() {
                        $('#mws-enroll-result').html('<span style="color:#d63638;">خطا در ارتباط</span>');
                        $btn.prop('disabled', false).text('اتصال به دوره');
                    }
                });
            });
        });
        </script>
        <?php
    }

    /* ------------------------------------------------------------------ */
    /*  AJAX: دریافت لیست دانشجویان                                        */
    /* ------------------------------------------------------------------ */

    public function ajax_get_course_students() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }

        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        if (!$course_id) {
            wp_send_json_error(array('message' => 'شناسه دوره نامعتبر'));
        }

        if (!$this->moodle_api || !$this->moodle_api->is_configured()) {
            wp_send_json_error(array('message' => 'Moodle تنظیم نشده'));
        }

        $moodle_users = $this->moodle_api->get_course_enrolled_users($course_id);
        if ($moodle_users === false) {
            wp_send_json_error(array('message' => 'خطا در دریافت لیست از Moodle'));
        }
        if (!is_array($moodle_users)) {
            wp_send_json_success(array('students' => array()));
        }

        global $wpdb;

        // نقشه: moodle_user_id → wp_user_id
        $rows = $wpdb->get_results(
            "SELECT user_id, meta_value FROM {$wpdb->usermeta}
             WHERE meta_key = 'mws_moodle_user_id' AND meta_value != '' AND meta_value != '0'"
        );
        $moodle_to_wp = array();
        foreach ($rows as $row) {
            $moodle_to_wp[intval($row->meta_value)] = intval($row->user_id);
        }

        $now      = time();
        $students = array();

        foreach ($moodle_users as $mu) {
            $moodle_user_id = intval($mu['id']);
            $wp_user_id     = isset($moodle_to_wp[$moodle_user_id]) ? $moodle_to_wp[$moodle_user_id] : 0;

            $enrolled_at_j = '';
            $expires_at_j  = '';
            $expires_at_raw = '';
            $expired       = false;
            $status        = (!empty($mu['suspended'])) ? 'suspended' : 'active';

            if ($wp_user_id) {
                $wp_enrollments = get_user_meta($wp_user_id, 'mws_enrollments', true);
                if (is_array($wp_enrollments)) {
                    foreach ($wp_enrollments as $enr) {
                        if (intval($enr['course_id']) === $course_id) {
                            if (!empty($enr['enrolled_at'])) {
                                $enrolled_at_j = $this->to_jalali($enr['enrolled_at']);
                            }
                            if (!empty($enr['expires_at'])) {
                                $expires_at_raw = $enr['expires_at'];
                                $expires_at_j   = $this->to_jalali($enr['expires_at']);
                                $expired        = strtotime($enr['expires_at']) < $now;
                            }
                            if (!empty($enr['status']) && $enr['status'] === 'suspended') {
                                $status = 'suspended';
                            }
                            break;
                        }
                    }
                }
            }

            $fullname = isset($mu['fullname']) ? $mu['fullname']
                : (trim(($mu['firstname'] ?? '') . ' ' . ($mu['lastname'] ?? '')));

            $students[] = array(
                'moodle_user_id' => $moodle_user_id,
                'wp_user_id'     => $wp_user_id,
                'fullname'       => $fullname,
                'email'          => $mu['email'] ?? '',
                'enrolled_at'    => $enrolled_at_j,
                'expires_at'     => $expires_at_j,
                'expires_at_raw' => $expires_at_raw,
                'expired'        => $expired,
                'status'         => $status,
            );
        }

        wp_send_json_success(array('students' => $students));
    }

    /* ------------------------------------------------------------------ */
    /*  AJAX: نمره (lazy)                                                   */
    /* ------------------------------------------------------------------ */

    public function ajax_get_student_grade() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }

        $course_id      = isset($_POST['course_id'])      ? intval($_POST['course_id'])      : 0;
        $moodle_user_id = isset($_POST['moodle_user_id']) ? intval($_POST['moodle_user_id']) : 0;

        if (!$course_id || !$moodle_user_id) {
            wp_send_json_error(array('message' => 'پارامترهای نامعتبر'));
        }

        $grade = $this->fetch_grade($moodle_user_id, $course_id);
        wp_send_json_success(array('grade' => $grade));
    }

    private function fetch_grade($moodle_user_id, $course_id) {
        if (!$this->moodle_api || !$this->moodle_api->is_configured()) {
            return null;
        }

        // روش اول: grade_items
        $result = $this->moodle_api->get_user_grade_items($moodle_user_id, $course_id);
        if ($result && !is_wp_error($result) && !empty($result['usergrades'])) {
            foreach ($result['usergrades'][0]['gradeitems'] ?? array() as $item) {
                if (isset($item['itemtype']) && $item['itemtype'] === 'course') {
                    $g = trim($item['gradeformatted'] ?? '');
                    if ($g !== '' && $g !== '-') return $g;
                    return null;
                }
            }
        }

        // روش دوم: grades_table
        $result = $this->moodle_api->get_user_grades($moodle_user_id, $course_id);
        if ($result && !is_wp_error($result) && !empty($result['tables'])) {
            foreach ($result['tables'][0]['tabledata'] ?? array() as $row) {
                $name = strip_tags($row['itemname']['content'] ?? '');
                if (stripos($name, 'Course total') !== false || stripos($name, 'جمع کل') !== false) {
                    $g = trim(strip_tags($row['grade']['content'] ?? ''));
                    if ($g !== '' && $g !== '-') return $g;
                    return null;
                }
            }
        }

        return null;
    }

    /* ------------------------------------------------------------------ */
    /*  AJAX: اتصال کاربر به دوره                                          */
    /* ------------------------------------------------------------------ */

    public function ajax_enroll_user() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }

        $course_id = isset($_POST['course_id']) ? intval($_POST['course_id']) : 0;
        $user_id   = isset($_POST['user_id'])   ? intval($_POST['user_id'])   : 0;

        if (!$course_id || !$user_id) {
            wp_send_json_error(array('message' => 'پارامترهای نامعتبر'));
        }

        if (!class_exists('MWS_Enrollment')) {
            wp_send_json_error(array('message' => 'کلاس Enrollment در دسترس نیست'));
        }

        $enrollment = new MWS_Enrollment();
        $result     = $enrollment->enroll_user($user_id, $course_id);

        if ($result) {
            wp_send_json_success(array('message' => 'کاربر با موفقیت به دوره متصل شد'));
        } else {
            wp_send_json_error(array('message' => 'خطا در اتصال کاربر به دوره'));
        }
    }

    /* ------------------------------------------------------------------ */
    /*  AJAX: قطع اتصال                                                    */
    /* ------------------------------------------------------------------ */

    public function ajax_unenroll_user() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }

        $course_id      = isset($_POST['course_id'])      ? intval($_POST['course_id'])      : 0;
        $moodle_user_id = isset($_POST['moodle_user_id']) ? intval($_POST['moodle_user_id']) : 0;
        $wp_user_id     = isset($_POST['wp_user_id'])     ? intval($_POST['wp_user_id'])     : 0;

        if (!$course_id || !$moodle_user_id) {
            wp_send_json_error(array('message' => 'پارامترهای نامعتبر'));
        }

        $result = $this->moodle_api->unenroll_user($moodle_user_id, $course_id);

        if ($wp_user_id) {
            $enrollments = get_user_meta($wp_user_id, 'mws_enrollments', true);
            if (is_array($enrollments)) {
                $enrollments = array_values(array_filter($enrollments, function ($e) use ($course_id) {
                    return intval($e['course_id']) !== $course_id;
                }));
                update_user_meta($wp_user_id, 'mws_enrollments', $enrollments);
            }
        }

        if ($result && !is_wp_error($result)) {
            wp_send_json_success(array('message' => 'اتصال با موفقیت قطع شد'));
        } else {
            wp_send_json_error(array('message' => 'خطا در قطع اتصال از Moodle'));
        }
    }

    /* ------------------------------------------------------------------ */
    /*  AJAX: تغییر تاریخ پایان                                            */
    /* ------------------------------------------------------------------ */

    public function ajax_update_expiry() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }

        $course_id      = isset($_POST['course_id'])      ? intval($_POST['course_id'])      : 0;
        $moodle_user_id = isset($_POST['moodle_user_id']) ? intval($_POST['moodle_user_id']) : 0;
        $wp_user_id     = isset($_POST['wp_user_id'])     ? intval($_POST['wp_user_id'])     : 0;
        $days           = isset($_POST['days'])           ? intval($_POST['days'])           : 0;

        if (!$course_id || !$moodle_user_id || $days === 0) {
            wp_send_json_error(array('message' => 'پارامترهای نامعتبر'));
        }

        $new_timeend      = 0;
        $new_expires      = null;
        $new_expires_j    = null;

        if ($wp_user_id) {
            $enrollments = get_user_meta($wp_user_id, 'mws_enrollments', true);
            if (!is_array($enrollments)) $enrollments = array();

            $found = false;
            foreach ($enrollments as &$enr) {
                if (intval($enr['course_id']) === $course_id) {
                    $base        = !empty($enr['expires_at']) ? strtotime($enr['expires_at']) : time();
                    $new_timeend = $base + ($days * DAY_IN_SECONDS);
                    $new_expires = $new_timeend > time() ? date('Y-m-d H:i:s', $new_timeend) : null;
                    if ($new_expires) {
                        $enr['expires_at'] = $new_expires;
                    } else {
                        unset($enr['expires_at']);
                    }
                    $found = true;
                    break;
                }
            }
            unset($enr);

            if (!$found) {
                $new_timeend = time() + ($days * DAY_IN_SECONDS);
                $new_expires = date('Y-m-d H:i:s', $new_timeend);
                $enrollments[] = array(
                    'course_id'   => $course_id,
                    'enrolled_at' => current_time('mysql'),
                    'expires_at'  => $new_expires,
                    'status'      => 'active',
                );
            }

            update_user_meta($wp_user_id, 'mws_enrollments', $enrollments);
        } else {
            $new_timeend = time() + ($days * DAY_IN_SECONDS);
            $new_expires = date('Y-m-d H:i:s', $new_timeend);
        }

        $this->moodle_api->enroll_user($moodle_user_id, $course_id, 5, 0, max(0, $new_timeend));

        if ($new_expires) {
            $new_expires_j = $this->to_jalali($new_expires);
        }

        wp_send_json_success(array(
            'message'        => 'تاریخ پایان به‌روز شد',
            'expires_at'     => $new_expires_j,
            'expires_at_raw' => $new_expires,
        ));
    }

    /* ------------------------------------------------------------------ */
    /*  نمایش دوره‌ها در پروفایل کاربر                                      */
    /* ------------------------------------------------------------------ */

    public function user_courses_profile_section($user) {
        if (!current_user_can('manage_options')) return;

        $enrollments    = get_user_meta($user->ID, 'mws_enrollments', true);
        $moodle_user_id = get_user_meta($user->ID, 'mws_moodle_user_id', true);
        $nonce          = wp_create_nonce('mws_ajax_nonce');
        $now            = time();
        ?>
        <div id="mws-user-courses" style="margin-top:30px;padding-top:20px;border-top:1px solid #ddd;">
            <h2>دوره‌های Moodle</h2>

            <?php if ($moodle_user_id) : ?>
            <p style="margin-bottom:12px;">
                <strong>شناسه Moodle:</strong> <?php echo esc_html($moodle_user_id); ?>
                &nbsp;|&nbsp;
                <button type="button"
                        id="mws-sync-courses-btn"
                        class="button"
                        data-user-id="<?php echo esc_attr($user->ID); ?>"
                        data-nonce="<?php echo esc_attr($nonce); ?>">
                    سینک دوره‌ها از Moodle
                </button>
                <span id="mws-sync-status" style="margin-right:10px;font-size:13px;"></span>
            </p>
            <?php endif; ?>

            <?php if (is_array($enrollments) && !empty($enrollments)) : ?>
            <table class="wp-list-table widefat fixed striped" style="max-width:800px;">
                <thead>
                    <tr>
                        <th>نام دوره</th>
                        <th>تاریخ ثبت‌نام</th>
                        <th>تاریخ پایان</th>
                        <th>وضعیت</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($enrollments as $enr) :
                        $moodle_cid  = intval($enr['course_id']);
                        $wp_courses  = get_posts(array(
                            'post_type'      => 'moodle_course',
                            'posts_per_page' => 1,
                            'fields'         => 'ids',
                            'meta_query'     => array(array(
                                'key'   => 'mws_moodle_course_id',
                                'value' => $moodle_cid,
                            )),
                        ));
                        $course_name = !empty($wp_courses)
                            ? get_the_title($wp_courses[0])
                            : ('دوره Moodle #' . $moodle_cid);

                        $enrolled_j = !empty($enr['enrolled_at']) ? $this->to_jalali($enr['enrolled_at']) : 'نامشخص';
                        $expires_j  = !empty($enr['expires_at'])  ? $this->to_jalali($enr['expires_at'])  : '';
                        $is_expired = !empty($enr['expires_at']) && strtotime($enr['expires_at']) < $now;
                        $status     = isset($enr['status']) ? $enr['status'] : 'active';
                    ?>
                    <tr>
                        <td><?php echo esc_html($course_name); ?></td>
                        <td><?php echo esc_html($enrolled_j); ?></td>
                        <td>
                            <?php if (!empty($enr['expires_at'])) : ?>
                                <span style="color:<?php echo $is_expired ? '#d63638' : '#00a32a'; ?>">
                                    <?php echo esc_html($expires_j); ?><?php echo $is_expired ? ' ⚠' : ''; ?>
                                </span>
                            <?php else : ?>
                                <span style="color:#999;">نامحدود</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ($status === 'active') : ?>
                                <span style="color:#00a32a;font-weight:600;">فعال</span>
                            <?php else : ?>
                                <span style="color:#d63638;font-weight:600;">تعلیق</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php else : ?>
            <p style="color:#666;">
                هیچ دوره‌ای ثبت نشده است.
                <?php if ($moodle_user_id) : ?>
                    از دکمه «سینک دوره‌ها» برای بارگذاری اطلاعات از Moodle استفاده کنید.
                <?php endif; ?>
            </p>
            <?php endif; ?>
        </div>

        <script>
        jQuery(document).ready(function($) {
            $('#mws-sync-courses-btn').on('click', function() {
                var $btn  = $(this);
                var uid   = $btn.data('user-id');
                var nonce = $btn.data('nonce');
                $btn.prop('disabled', true).text('در حال سینک...');
                $('#mws-sync-status').html('<span class="spinner is-active" style="float:none;"></span>');
                $.ajax({
                    url: ajaxurl, type: 'POST',
                    data: { action: 'mws_sync_user_enrollments_to_meta', user_id: uid, nonce: nonce },
                    success: function(res) {
                        if (res.success) {
                            $('#mws-sync-status').html('<span style="color:#00a32a;">✓ ' + res.data.message + ' — در حال بارگذاری...</span>');
                            setTimeout(function() { location.reload(); }, 1500);
                        } else {
                            $('#mws-sync-status').html('<span style="color:#d63638;">✗ ' + (res.data.message || 'خطا') + '</span>');
                            $btn.prop('disabled', false).text('سینک دوره‌ها از Moodle');
                        }
                    },
                    error: function() {
                        $('#mws-sync-status').html('<span style="color:#d63638;">خطا در ارتباط</span>');
                        $btn.prop('disabled', false).text('سینک دوره‌ها از Moodle');
                    }
                });
            });
        });
        </script>
        <?php
    }

    /* ------------------------------------------------------------------ */
    /*  AJAX: سینک دوره‌های کاربر به WP meta                               */
    /* ------------------------------------------------------------------ */

    public function ajax_sync_user_enrollments() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }

        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        if (!$user_id) {
            wp_send_json_error(array('message' => 'شناسه کاربر نامعتبر'));
        }

        $moodle_user_id = get_user_meta($user_id, 'mws_moodle_user_id', true);
        if (!$moodle_user_id) {
            wp_send_json_error(array('message' => 'کاربر هنوز با Moodle سینک نشده است'));
        }

        if (!$this->moodle_api || !$this->moodle_api->is_configured()) {
            wp_send_json_error(array('message' => 'Moodle تنظیم نشده'));
        }

        $moodle_courses = $this->moodle_api->get_user_enrollments($moodle_user_id);
        if (!$moodle_courses || is_wp_error($moodle_courses)) {
            wp_send_json_error(array('message' => 'خطا در دریافت دوره‌ها از Moodle. شاید این کاربر دوره فعالی ندارد.'));
        }

        $existing = get_user_meta($user_id, 'mws_enrollments', true);
        if (!is_array($existing)) $existing = array();

        $existing_map = array();
        foreach ($existing as $enr) {
            $existing_map[intval($enr['course_id'])] = true;
        }

        $added = 0;
        foreach ((array) $moodle_courses as $mc) {
            $cid = intval($mc['id']);
            if (!isset($existing_map[$cid])) {
                $enrolled_at = !empty($mc['lastaccess']) && intval($mc['lastaccess']) > 0
                    ? date('Y-m-d H:i:s', intval($mc['lastaccess']))
                    : current_time('mysql');
                $existing[] = array(
                    'course_id'   => $cid,
                    'enrolled_at' => $enrolled_at,
                    'status'      => 'active',
                );
                $added++;
            }
        }

        update_user_meta($user_id, 'mws_enrollments', $existing);

        wp_send_json_success(array(
            'message' => $added . ' دوره جدید به لیست اضافه شد',
        ));
    }
}
