<?php
/**
 * کلاس همگام‌سازی دسته‌بندی‌ها بین Moodle و WordPress
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_Category_Sync {
    private $moodle_api;
    
    public function __construct() {
        $this->moodle_api = new MWS_Moodle_API();
    }
    
    /**
     * راه‌اندازی هوک‌ها
     */
    public function init() {
        // ثبت Taxonomy برای دسته‌بندی‌های Moodle
        add_action('init', array($this, 'register_taxonomy'));
        
        // همگام‌سازی دسته‌بندی‌ها از صفحه ادمین
        add_action('wp_ajax_mws_sync_categories', array($this, 'ajax_sync_categories'));
        add_action('wp_ajax_mws_sync_single_category', array($this, 'ajax_sync_single_category'));
        
        // افزودن ستون‌های سفارشی
        add_filter('manage_edit-moodle_category_columns', array($this, 'add_category_columns'));
        add_filter('manage_moodle_category_custom_column', array($this, 'render_category_columns'), 10, 3);
        
        // همگام‌سازی هنگام ذخیره دسته‌بندی
        add_action('created_moodle_category', array($this, 'sync_to_moodle_on_create'), 10, 2);
        add_action('edited_moodle_category', array($this, 'sync_to_moodle_on_edit'), 10, 2);
        
        // افزودن فیلدهای سفارشی به فرم دسته‌بندی
        add_action('moodle_category_add_form_fields', array($this, 'add_category_fields'));
        add_action('moodle_category_edit_form_fields', array($this, 'edit_category_fields'));
        add_action('created_moodle_category', array($this, 'save_category_fields'));
        add_action('edited_moodle_category', array($this, 'save_category_fields'));
    }
    
    /**
     * ثبت Taxonomy برای دسته‌بندی‌های Moodle
     */
    public function register_taxonomy() {
        $labels = array(
            'name' => 'دسته‌بندی‌های Moodle',
            'singular_name' => 'دسته‌بندی Moodle',
            'menu_name' => 'دسته‌بندی‌های Moodle',
            'all_items' => 'همه دسته‌بندی‌ها',
            'edit_item' => 'ویرایش دسته‌بندی',
            'view_item' => 'مشاهده دسته‌بندی',
            'update_item' => 'به‌روزرسانی دسته‌بندی',
            'add_new_item' => 'افزودن دسته‌بندی جدید',
            'new_item_name' => 'نام دسته‌بندی جدید',
            'search_items' => 'جستجوی دسته‌بندی‌ها'
        );
        
        register_taxonomy('moodle_category', array('moodle_course'), array(
            'labels' => $labels,
            'hierarchical' => true,
            'public' => true,
            'show_ui' => true,
            'show_admin_column' => true,
            'show_in_rest' => true,
            'rewrite' => array('slug' => 'moodle-category'),
            'show_in_menu' => false // مخفی کردن منوی جداگانه
        ));
    }
    
    /**
     * همگام‌سازی دسته‌بندی‌ها از Moodle
     */
    public function sync_categories() {
        if (!$this->moodle_api->is_configured()) {
            return array('success' => false, 'message' => 'Moodle تنظیم نشده است');
        }
        
        $categories = $this->moodle_api->get_categories();
        
        if (is_wp_error($categories)) {
            return array('success' => false, 'message' => $categories->get_error_message());
        }
        
        if (!$categories || !is_array($categories)) {
            return array('success' => false, 'message' => 'هیچ دسته‌بندی‌ای یافت نشد');
        }
        
        $synced = 0;
        $updated = 0;
        
        foreach ($categories as $category) {
            // رد کردن دسته‌بندی پیش‌فرض (id = 0)
            if ($category['id'] == 0) {
                continue;
            }
            
            $result = $this->create_or_update_category($category);
            
            if ($result['action'] == 'created') {
                $synced++;
            } elseif ($result['action'] == 'updated') {
                $updated++;
            }
        }
        
        return array(
            'success' => true,
            'message' => sprintf('همگام‌سازی کامل شد. جدید: %d، به‌روزرسانی: %d', $synced, $updated),
            'synced' => $synced,
            'updated' => $updated
        );
    }
    
    /**
     * ایجاد یا به‌روزرسانی دسته‌بندی در WordPress
     */
    private function create_or_update_category($category) {
        // جستجوی دسته‌بندی موجود
        $terms = get_terms(array(
            'taxonomy' => 'moodle_category',
            'hide_empty' => false,
            'meta_query' => array(
                array(
                    'key' => 'mws_moodle_category_id',
                    'value' => $category['id'],
                    'compare' => '='
                )
            )
        ));
        
        $term_data = array(
            'description' => isset($category['description']) ? wp_kses_post($category['description']) : ''
        );
        
        // پیدا کردن والد
        if (isset($category['parent']) && $category['parent'] > 0) {
            $parent_terms = get_terms(array(
                'taxonomy' => 'moodle_category',
                'hide_empty' => false,
                'meta_query' => array(
                    array(
                        'key' => 'mws_moodle_category_id',
                        'value' => $category['parent'],
                        'compare' => '='
                    )
                )
            ));
            
            if (!empty($parent_terms)) {
                $term_data['parent'] = $parent_terms[0]->term_id;
            }
        }
        
        if (!empty($terms)) {
            // به‌روزرسانی دسته‌بندی موجود
            $term = wp_update_term($terms[0]->term_id, 'moodle_category', array(
                'name' => sanitize_text_field($category['name']),
                'description' => $term_data['description'],
                'parent' => isset($term_data['parent']) ? $term_data['parent'] : 0
            ));
            
            if (!is_wp_error($term)) {
                update_term_meta($terms[0]->term_id, 'mws_synced_at', current_time('mysql'));
                return array('success' => true, 'action' => 'updated', 'term_id' => $term['term_id']);
            }
        } else {
            // ایجاد دسته‌بندی جدید
            $term = wp_insert_term(
                sanitize_text_field($category['name']),
                'moodle_category',
                $term_data
            );
            
            if (!is_wp_error($term)) {
                update_term_meta($term['term_id'], 'mws_moodle_category_id', intval($category['id']));
                update_term_meta($term['term_id'], 'mws_synced_at', current_time('mysql'));
                return array('success' => true, 'action' => 'created', 'term_id' => $term['term_id']);
            }
        }
        
        return array('success' => false, 'action' => 'failed');
    }
    
    /**
     * افزودن ستون‌های سفارشی
     */
    public function add_category_columns($columns) {
        $new_columns = array();
        $new_columns['cb'] = $columns['cb'];
        $new_columns['name'] = 'نام';
        $new_columns['mws_moodle_id'] = 'شناسه Moodle';
        $new_columns['mws_sync'] = 'سینک دستی';
        $new_columns['posts'] = $columns['posts'];
        
        return $new_columns;
    }
    
    /**
     * نمایش محتوای ستون‌های سفارشی
     */
    public function render_category_columns($content, $column_name, $term_id) {
        switch ($column_name) {
            case 'mws_moodle_id':
                $moodle_id = get_term_meta($term_id, 'mws_moodle_category_id', true);
                if ($moodle_id) {
                    echo '<strong>' . esc_html($moodle_id) . '</strong>';
                } else {
                    echo '<span style="color: #999;">هنوز سینک نشده</span>';
                }
                break;
                
            case 'mws_sync':
                $moodle_id = get_term_meta($term_id, 'mws_moodle_category_id', true);
                echo '<button type="button" class="button button-small mws-sync-single-category" data-term-id="' . esc_attr($term_id) . '">';
                echo $moodle_id ? 'به‌روزرسانی' : 'سینک به Moodle';
                echo '</button>';
                echo '<span class="mws-sync-status-' . esc_attr($term_id) . '" style="margin-right: 5px;"></span>';
                break;
        }
        
        return $content;
    }
    
    /**
     * افزودن فیلدهای سفارشی به فرم افزودن دسته‌بندی
     */
    public function add_category_fields() {
        ?>
        <div class="form-field">
            <label>
                <input type="checkbox" name="mws_sync_to_moodle" value="1" />
                همگام‌سازی با Moodle
            </label>
            <p class="description">با فعال کردن این گزینه، دسته‌بندی در Moodle نیز ایجاد می‌شود</p>
        </div>
        <?php
    }
    
    /**
     * افزودن فیلدهای سفارشی به فرم ویرایش دسته‌بندی
     */
    public function edit_category_fields($term) {
        $moodle_id = get_term_meta($term->term_id, 'mws_moodle_category_id', true);
        $synced_at = get_term_meta($term->term_id, 'mws_synced_at', true);
        ?>
        <tr class="form-field">
            <th scope="row">
                <label>شناسه Moodle</label>
            </th>
            <td>
                <strong><?php echo esc_html($moodle_id ?: 'هنوز سینک نشده'); ?></strong>
            </td>
        </tr>
        <tr class="form-field">
            <th scope="row">
                <label>آخرین همگام‌سازی</label>
            </th>
            <td>
                <?php echo esc_html($synced_at ?: '-'); ?>
            </td>
        </tr>
        <tr class="form-field">
            <th scope="row">
                <label>همگام‌سازی با Moodle</label>
            </th>
            <td>
                <label>
                    <input type="checkbox" name="mws_sync_to_moodle" value="1" />
                    همگام‌سازی با Moodle هنگام ذخیره
                </label>
            </td>
        </tr>
        <?php
    }
    
    /**
     * ذخیره فیلدهای سفارشی
     */
    public function save_category_fields($term_id) {
        if (isset($_POST['mws_sync_to_moodle']) && $_POST['mws_sync_to_moodle'] == '1') {
            $this->create_category_in_moodle($term_id);
        }
    }
    
    /**
     * همگام‌سازی هنگام ایجاد دسته‌بندی
     */
    public function sync_to_moodle_on_create($term_id, $tt_id) {
        if (isset($_POST['mws_sync_to_moodle']) && $_POST['mws_sync_to_moodle'] == '1') {
            $this->create_category_in_moodle($term_id);
        }
    }
    
    /**
     * همگام‌سازی هنگام ویرایش دسته‌بندی
     */
    public function sync_to_moodle_on_edit($term_id, $tt_id) {
        if (isset($_POST['mws_sync_to_moodle']) && $_POST['mws_sync_to_moodle'] == '1') {
            $this->create_category_in_moodle($term_id);
        }
    }
    
    /**
     * ایجاد یا به‌روزرسانی دسته‌بندی در Moodle
     */
    public function create_category_in_moodle($term_id) {
        if (!$this->moodle_api->is_configured()) {
            return false;
        }
        
        $term = get_term($term_id);
        if (!$term || $term->taxonomy != 'moodle_category') {
            return false;
        }
        
        $moodle_id = get_term_meta($term_id, 'mws_moodle_category_id', true);
        
        // پیدا کردن والد در Moodle
        $parent_moodle_id = 0;
        if ($term->parent > 0) {
            $parent_moodle_id = get_term_meta($term->parent, 'mws_moodle_category_id', true);
            if (!$parent_moodle_id) {
                $parent_moodle_id = 0;
            }
        }
        
        $category_data = array(
            'name' => $term->name,
            'parent' => intval($parent_moodle_id),
            'description' => $term->description
        );
        
        if ($moodle_id) {
            // به‌روزرسانی دسته‌بندی موجود
            $category_data['id'] = intval($moodle_id);
            $result = $this->moodle_api->update_category($category_data);
        } else {
            // ایجاد دسته‌بندی جدید
            $result = $this->moodle_api->create_category($category_data);
        }
        
        if (!is_wp_error($result)) {
            if (!$moodle_id && isset($result[0]['id'])) {
                update_term_meta($term_id, 'mws_moodle_category_id', intval($result[0]['id']));
            }
            update_term_meta($term_id, 'mws_synced_at', current_time('mysql'));
            return true;
        }
        
        return false;
    }
    
    /**
     * همگام‌سازی دسته‌بندی‌ها (AJAX)
     */
    public function ajax_sync_categories() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $result = $this->sync_categories();
        
        if ($result['success']) {
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    /**
     * همگام‌سازی یک دسته‌بندی (AJAX)
     */
    public function ajax_sync_single_category() {
        check_ajax_referer('mws_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'دسترسی مجاز نیست'));
        }
        
        $term_id = isset($_POST['term_id']) ? intval($_POST['term_id']) : 0;
        
        if (!$term_id) {
            wp_send_json_error(array('message' => 'شناسه دسته‌بندی نامعتبر است'));
        }
        
        $result = $this->create_category_in_moodle($term_id);
        
        if ($result) {
            $moodle_id = get_term_meta($term_id, 'mws_moodle_category_id', true);
            wp_send_json_success(array(
                'message' => 'همگام‌سازی با موفقیت انجام شد',
                'moodle_id' => $moodle_id
            ));
        } else {
            wp_send_json_error(array('message' => 'همگام‌سازی با خطا مواجه شد'));
        }
    }
}

