<?php
/**
 * کلاس مدیریت صفحه ادمین و تنظیمات
 */

if (!defined('ABSPATH')) {
    exit;
}

class MWS_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_notices', array($this, 'show_admin_notices'));
        add_action('admin_init', array($this, 'handle_sso_file_download'));
    }
    
    /**
     * مدیریت دانلود فایل SSO
     */
    public function handle_sso_file_download() {
        if (!isset($_GET['mws_download_sso_file']) || !current_user_can('manage_options')) {
            return;
        }
        
        // بررسی nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'mws_download_sso_file')) {
            wp_die('خطای امنیتی: درخواست نامعتبر است.');
        }
        
        // دریافت مقادیر
        $wp_url = home_url();
        $sso_secret = get_option('mws_sso_secret_key', '');
        
        if (empty($sso_secret)) {
            $sso_secret = wp_generate_password(64, true, true);
            update_option('mws_sso_secret_key', $sso_secret);
        }
        
        // خواندن فایل template
        $template_file = MWS_PLUGIN_DIR . 'mws-sso.php';
        if (!file_exists($template_file)) {
            wp_die('فایل SSO پیدا نشد. لطفاً مطمئن شوید که فایل mws-sso.php در پوشه پلاگین وجود دارد.');
        }
        
        $file_content = file_get_contents($template_file);
        
        // جایگزینی مقادیر - استفاده از regex برای اطمینان از جایگزینی صحیح
        $file_content = preg_replace(
            "/define\('WP_URL',\s*'[^']*'\);/",
            "define('WP_URL', '" . esc_attr($wp_url) . "');",
            $file_content
        );
        $file_content = preg_replace(
            "/define\('JWT_SECRET',\s*'[^']*'\);/",
            "define('JWT_SECRET', '" . esc_attr($sso_secret) . "');",
            $file_content
        );
        
        // کلید راه‌اندازی برای ساخت خودکار توکن در Moodle
        $setup_key = get_option('mws_sso_setup_key', '');
        if (empty($setup_key)) {
            $setup_key = wp_generate_password(32, true, true);
            update_option('mws_sso_setup_key', $setup_key);
        }
        $file_content = preg_replace(
            "/define\('MWS_SETUP_KEY',\s*'[^']*'\);/",
            "define('MWS_SETUP_KEY', '" . esc_attr($setup_key) . "');",
            $file_content
        );
        
        // ارسال فایل برای دانلود
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="mws-sso.php"');
        header('Content-Length: ' . strlen($file_content));
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');
        
        echo $file_content;
        exit;
    }
    
    /**
     * افزودن منوی ادمین
     */
    public function add_admin_menu() {
        add_menu_page(
            'سینک مودل رویال',
            'سینک مودل رویال',
            'manage_options',
            'royal-moodle-sync',
            array($this, 'settings_page'),
            'dashicons-update',
            30
        );
        
        add_submenu_page(
            'royal-moodle-sync',
            'تنظیمات',
            'تنظیمات',
            'manage_options',
            'royal-moodle-sync',
            array($this, 'settings_page')
        );
        
        // منوی دوره‌ها
        add_submenu_page(
            'royal-moodle-sync',
            'دوره‌ها',
            'دوره‌ها',
            'manage_options',
            'edit.php?post_type=moodle_course'
        );
        
        // منوی افزودن دوره
        add_submenu_page(
            'royal-moodle-sync',
            'افزودن دوره',
            'افزودن دوره',
            'manage_options',
            'post-new.php?post_type=moodle_course'
        );
        
        // منوی دسته‌بندی‌ها
        add_submenu_page(
            'royal-moodle-sync',
            'دسته‌بندی‌ها',
            'دسته‌بندی‌ها',
            'manage_options',
            'edit-tags.php?taxonomy=moodle_category'
        );
        
        // منوی سفارشات دوره‌ها
        if (class_exists('WooCommerce')) {
            add_submenu_page(
                'royal-moodle-sync',
                'سفارشات دوره‌ها',
                'سفارشات دوره‌ها',
                'manage_options',
                'royal-moodle-sync-orders',
                array($this, 'orders_page')
            );
        }
        
    }
    
    /**
     * صفحه سفارشات (از class-orders)
     */
    public function orders_page() {
        if (!class_exists('MWS_Orders')) {
            echo '<div class="wrap"><p>کلاس سفارشات در دسترس نیست.</p></div>';
            return;
        }
        
        $orders = new MWS_Orders();
        if (method_exists($orders, 'orders_page')) {
            $orders->orders_page();
        } else {
            // اگر متد orders_page وجود ندارد، از متد اصلی استفاده می‌کنیم
            $orders->display_orders_list();
        }
    }
    
    /**
     * صفحه اتصال کاربران (از class-user-enrollment)
     */
    public function enrollment_page() {
        if (!class_exists('MWS_User_Enrollment_Manager')) {
            echo '<div class="wrap"><p>کلاس مدیریت اتصال کاربران در دسترس نیست.</p></div>';
            return;
        }
        
        $enrollment_manager = new MWS_User_Enrollment_Manager();
        if (method_exists($enrollment_manager, 'enrollment_page')) {
            $enrollment_manager->enrollment_page();
        }
    }
    
    /**
     * ثبت تنظیمات
     */
    public function register_settings() {
        register_setting('mws_settings', 'mws_moodle_url', array(
            'sanitize_callback' => 'esc_url_raw'
        ));
        register_setting('mws_settings', 'mws_moodle_token', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('mws_settings', 'mws_auto_sync_users', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('mws_settings', 'mws_auto_sync_courses', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('mws_settings', 'mws_sso_token_expiry', array(
            'sanitize_callback' => 'absint',
            'default' => 3600
        ));
        register_setting('mws_settings', 'mws_sso_token_unlimited', array(
            'sanitize_callback' => 'sanitize_text_field'
        ));
        register_setting('mws_settings', 'mws_sync_batch_size', array(
            'sanitize_callback' => 'absint',
            'default' => 100
        ));
    }
    
    /**
     * نمایش اعلان‌های ادمین
     */
    public function show_admin_notices() {
        $screen = get_current_screen();
        if ($screen && (strpos($screen->id, 'royal-moodle-sync') !== false || strpos($screen->id, 'moodle-wp-sync') !== false)) {
            $moodle_url = get_option('mws_moodle_url');
            $moodle_token = get_option('mws_moodle_token');
            
            if (empty($moodle_url) || empty($moodle_token)) {
                echo '<div class="notice notice-warning"><p>';
                echo '<strong>توجه:</strong> لطفاً آدرس Moodle و Token را در تنظیمات وارد کنید.';
                echo '</p></div>';
            }
        }
    }
    
    /**
     * صفحه تنظیمات
     */
    public function settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // تست اتصال
        $test_result = null;
        if (isset($_POST['test_connection'])) {
            check_admin_referer('mws_test_connection');
            $moodle_api = new MWS_Moodle_API();
            $test_result = $moodle_api->test_connection();
        }
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="mws-settings-container">
                <div class="mws-settings-main">
                    <form method="post" action="options.php">
                        <?php settings_fields('mws_settings'); ?>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="mws_moodle_url">آدرس Moodle</label>
                                </th>
                                <td>
                                    <input type="url" 
                                           id="mws_moodle_url" 
                                           name="mws_moodle_url" 
                                           value="<?php echo esc_attr(get_option('mws_moodle_url')); ?>" 
                                           class="regular-text" 
                                           placeholder="https://your-moodle-site.com" />
                                    <p class="description">
                                        آدرس کامل سایت Moodle شما (بدون اسلش در انتها)
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="mws_moodle_token">Web Service Token</label>
                                </th>
                                <td>
                                    <input type="text" 
                                           id="mws_moodle_token" 
                                           name="mws_moodle_token" 
                                           value="<?php echo esc_attr(get_option('mws_moodle_token')); ?>" 
                                           class="regular-text" />
                                    <p class="description">
                                        توکن Web Service که از Moodle دریافت کرده‌اید<br>
                                        <strong>راهنمای دریافت توکن:</strong><br>
                                        1. در Moodle به Site administration > Plugins > Web services > Manage tokens بروید<br>
                                        2. یک توکن جدید برای کاربر خود ایجاد کنید<br>
                                        3. توکن را کپی کرده و در اینجا وارد کنید
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">همگام‌سازی خودکار</th>
                                <td>
                                    <fieldset>
                                        <label>
                                            <input type="checkbox" 
                                                   name="mws_auto_sync_users" 
                                                   value="1" 
                                                   <?php checked(get_option('mws_auto_sync_users'), '1'); ?> />
                                            همگام‌سازی خودکار کاربران جدید
                                        </label>
                                        <br>
                                        <label>
                                            <input type="checkbox" 
                                                   name="mws_auto_sync_courses" 
                                                   value="1" 
                                                   <?php checked(get_option('mws_auto_sync_courses'), '1'); ?> />
                                            همگام‌سازی خودکار دوره‌ها (هر ساعت)
                                        </label>
                                    </fieldset>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label for="mws_sync_batch_size">حداکثر کاربر در هر بار همگام‌سازی</label>
                                </th>
                                <td>
                                    <?php $batch_size = get_option('mws_sync_batch_size', 100); ?>
                                    <input type="number" 
                                           id="mws_sync_batch_size" 
                                           name="mws_sync_batch_size" 
                                           value="<?php echo esc_attr($batch_size); ?>" 
                                           min="10" 
                                           max="500" 
                                           step="10" 
                                           class="small-text" />
                                    <p class="description">
                                        در هر بار زدن دکمه «همگام‌سازی کاربران»، حداکثر چند کاربر پردازش شود (پیشنهاد: ۵۰ تا ۲۰۰ برای جلوگیری از تایم‌اوت).
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">
                                    <label>مدت اعتبار توکن SSO</label>
                                </th>
                                <td>
                                    <?php
                                    $token_expiry = get_option('mws_sso_token_expiry', 3600);
                                    $token_unlimited = get_option('mws_sso_token_unlimited', '0');
                                    ?>
                                    <fieldset>
                                        <label>
                                            <input type="checkbox" 
                                                   name="mws_sso_token_unlimited" 
                                                   value="1" 
                                                   id="mws_sso_token_unlimited"
                                                   <?php checked($token_unlimited, '1'); ?>
                                                   onchange="document.getElementById('mws_sso_token_expiry').disabled = this.checked;" />
                                            مدت اعتبار نامحدود
                                        </label>
                                        <br><br>
                                        <label>
                                            مدت اعتبار (ثانیه):
                                            <input type="number" 
                                                   id="mws_sso_token_expiry"
                                                   name="mws_sso_token_expiry" 
                                                   value="<?php echo esc_attr($token_expiry); ?>" 
                                                   min="60"
                                                   step="60"
                                                   class="small-text"
                                                   <?php echo $token_unlimited == '1' ? 'disabled' : ''; ?> />
                                        </label>
                                        <p class="description">
                                            مدت زمان اعتبار توکن SSO به ثانیه. پیش‌فرض: 3600 ثانیه (1 ساعت)<br>
                                            <strong>مثال:</strong> 3600 = 1 ساعت، 86400 = 24 ساعت، 604800 = 7 روز<br>
                                            <strong>نکته:</strong> اگر "مدت اعتبار نامحدود" را فعال کنید، توکن هرگز منقضی نمی‌شود.
                                        </p>
                                    </fieldset>
                                </td>
                            </tr>
                        </table>
                        
                        <hr style="margin: 30px 0;">
                        
                        <h2 style="margin-top: 30px;">🔐 تنظیمات SSO (Single Sign-On)</h2>
                        <p class="description" style="margin-bottom: 20px;">
                            با فعال‌سازی SSO، کاربران WordPress می‌توانند به صورت خودکار به Moodle وارد شوند بدون نیاز به وارد کردن نام کاربری و رمز عبور.
                            </p>
                        <?php
                        $mws_setup_key = get_option('mws_sso_setup_key', '');
                        if (empty($mws_setup_key)) {
                            $mws_setup_key = wp_generate_password(32, true, true);
                            update_option('mws_sso_setup_key', $mws_setup_key);
                        }
                        ?>
                        <div style="background: #f0f6fc; border-right: 4px solid #0073aa; padding: 20px; margin: 20px 0; border-radius: 4px;">
                            <h3 style="margin-top: 0; color: #0073aa;">📥 دانلود فایل SSO برای Moodle</h3>
                            <p style="margin-bottom: 15px;">
                                فایل <code>mws-sso.php</code> را دانلود کنید که به صورت خودکار با تنظیمات شما (آدرس WordPress و Secret Key) پر شده است.
                            </p>
                            <p>
                                <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=royal-moodle-sync&mws_download_sso_file=1'), 'mws_download_sso_file'); ?>" 
                                   class="button button-primary button-large" 
                                   style="font-size: 14px; padding: 8px 20px; height: auto;">
                                    ⬇️ دانلود فایل mws-sso.php
                                </a>
                            </p>
                            <p style="margin-top: 15px; margin-bottom: 0; font-size: 13px; color: #666;">
                                <strong>نکته:</strong> بعد از دانلود، فایل را در root Moodle قرار دهید (همان مسیری که <code>config.php</code> است).
                            </p>
                            <table class="form-table" style="margin-top: 30px;">
                            <tr>
                                <th scope="row">
                                    <label>SSO Secret Key</label>
                                </th>
                                <td>
                                    <?php
                                    $sso_secret = get_option('mws_sso_secret_key', '');
                                    if (empty($sso_secret)) {
                                        // اگر Secret key وجود ندارد، یک عدد جدید ایجاد می‌کنیم
                                        $sso_secret = wp_generate_password(64, true, true);
                                        update_option('mws_sso_secret_key', $sso_secret);
                                    }
                                    ?>
                                    <div style="background: #f0f0f0; padding: 12px; border: 1px solid #ddd; border-radius: 4px; font-family: monospace; word-break: break-all; position: relative; padding-left: 80px;">
                                        <code id="mws-sso-secret-key" style="font-size: 13px; color: #333;"><?php echo esc_html($sso_secret); ?></code>
                                        <button type="button" 
                                                class="button button-small" 
                                                onclick="copySSOSecret()" 
                                                style="position: absolute; left: 10px; top: 10px;">
                                            📋 کپی
                                        </button>
                                    </div>
                                    <p class="description">
                                        این Secret key به صورت خودکار در فایل دانلود شده قرار می‌گیرد. در صورت نیاز می‌توانید آن را کپی کنید.
                                    </p>
                                    <script>
                                    function copySSOSecret() {
                                        var secretKey = document.getElementById('mws-sso-secret-key').textContent;
                                        navigator.clipboard.writeText(secretKey).then(function() {
                                            alert('Secret key کپی شد!');
                                        }, function() {
                                            // Fallback برای مرورگرهای قدیمی
                                            var textArea = document.createElement('textarea');
                                            textArea.value = secretKey;
                                            document.body.appendChild(textArea);
                                            textArea.select();
                                            document.execCommand('copy');
                                            document.body.removeChild(textArea);
                                            alert('Secret key کپی شد!');
                                        });
                                    }
                                    </script>
                                </td>
                            </tr>
                        </table>
                        </div>                     
                        <div style="background: #e8f5e9; border-right: 4px solid #4caf50; padding: 20px; margin: 20px 0; border-radius: 4px;">
                            <h3 style="margin-top: 0; color: #2e7d32;">🔑 ساخت خودکار توکن در Moodle (با فایل mws-sso.php)</h3>
                            <p style="margin-bottom: 15px;">
                                اگر فایل <code>mws-sso.php</code> را از همین صفحه دانلود کرده و در root مودل قرار داده‌اید، می‌توانید به‌جای ساخت دستی توکن، یک بار آدرس زیر را در مرورگر باز کنید تا سرویس و توکن با دسترسی‌های موردنیاز پلاگین ساخته شود.
                            </p>
                            <ol style="line-height: 2; padding-right: 20px; margin-bottom: 15px;">
                                <li>فایل <code>mws-sso.php</code> را از دکمهٔ بالا دانلود کنید و در <strong>root مودل</strong> (کنار <code>config.php</code>) قرار دهید.</li>
                                <li>در مودل با حساب <strong>مدیر سایت</strong> وارد شوید.</li>
                                <li>این آدرس را کپی کرده در مرورگر باز کنید (در صورت نیاز آدرس مودل را با آدرس سایت خود جایگزین کنید):</li>
                            </ol>
                            <p style="margin: 10px 0;">
                                <code style="background: #fff; padding: 10px 14px; display: block; margin: 8px 0; border-radius: 4px; word-break: break-all; border: 1px solid #c8e6c9;"><?php echo esc_html( get_option('mws_moodle_url') ? rtrim( get_option('mws_moodle_url'), '/' ) . '/mws-sso.php?mws_setup=1&amp;key=' . esc_attr( get_option('mws_sso_setup_key', '') ) : 'https://your-moodle.com/mws-sso.php?mws_setup=1&key=...' ); ?></code>
                            </p>
                            <ol start="4" style="line-height: 2; padding-right: 20px;">
                                <li>صفحه توکن را می‌بینید. توکن نمایش‌داده‌شده را کپی کنید و در فیلد «Web Service Token» بالای همین صفحه قرار دهید، سپس «ذخیره تنظیمات» را بزنید.</li>
                            </ol>
                            <p style="margin-bottom: 0; font-size: 13px; color: #555;">
                                <strong>نکته:</strong> اگر هنوز آدرس Moodle را ذخیره نکرده‌اید، ابتدا آدرس مودل را وارد و ذخیره کنید تا آدرس بالا به‌درستی نمایش داده شود. کلید <code>key</code> در همین آدرس نمایش داده می‌شود و به‌صورت خودکار در فایل دانلود شده هم قرار می‌گیرد.
                            </p>
                        </div>                                
                        <?php submit_button('ذخیره تنظیمات'); ?>
                    </form>
                    
                    <hr>
                    
                    <h2>تست اتصال</h2>
                    <form method="post" action="">
                        <?php wp_nonce_field('mws_test_connection'); ?>
                        <p>
                            <button type="submit" name="test_connection" class="button">
                                تست اتصال به Moodle
                            </button>
                        </p>
                    </form>
                    
                    <?php if ($test_result !== null) : ?>
                        <div class="notice notice-<?php echo $test_result['success'] ? 'success' : 'error'; ?>">
                            <p><?php echo esc_html($test_result['message']); ?></p>
                        </div>
                    <?php endif; ?>
                    
                    <hr>
                    
                    <h2>عملیات دستی</h2>
                    <div class="mws-manual-actions">
                        <p>
                            <button id="mws-sync-categories" class="button button-primary">
                                همگام‌سازی دسته‌بندی‌ها
                            </button>
                            <span class="mws-action-status" id="categories-status"></span>
                        </p>
                        <p>
                            <button id="mws-sync-courses" class="button button-primary">
                                همگام‌سازی دوره‌ها
                            </button>
                            <span class="mws-action-status" id="courses-status"></span>
                        </p>
                        <p>
                            <div id="mws-user-sync-stats" style="margin-bottom: 10px; padding: 10px; background: #f0f6fc; border-right: 3px solid #0073aa; border-radius: 4px; display: inline-block;">
                                <span class="mws-stats-total">تعداد کل کاربران: —</span><br>
                                <span class="mws-stats-synced">همگام‌سازی شده با Moodle: —</span>
                            </div>
                        </p>
                        <p>
                            <button id="mws-sync-users" class="button">
                                همگام‌سازی کاربران WordPress به Moodle
                            </button>
                            <span class="mws-action-status" id="users-status"></span>
                            <br>
                            <small style="color: #666; margin-top: 5px; display: block;">
                                در هر بار حداکثر <strong id="mws-batch-size-label"><?php echo esc_html(get_option('mws_sync_batch_size', 100)); ?></strong> کاربر همگام می‌شود. برای ادامه، دوباره دکمه را بزنید.
                            </small>
                        </p>
                        <p>
                            <button id="mws-import-users" class="button button-primary">
                                دریافت کاربران از Moodle
                            </button>
                            <span class="mws-action-status" id="import-users-status"></span>
                            <br>
                            <small style="color: #666; margin-top: 5px; display: block;">
                                این عملیات تمام کاربران Moodle را دریافت کرده و در WordPress ایجاد می‌کند. کاربرانی که قبلاً وجود دارند به‌روزرسانی می‌شوند.
                            </small>
                        </p>
                    </div>
                    
                    <div id="mws-ajax-result"></div>
                </div>
                
                <div class="mws-settings-sidebar">
                    <div class="mws-info-box">
                        <h3>راهنمای استفاده</h3>
                        <ol>
                            <li>آدرس Moodle و Token را وارد کنید</li>
                            <li>اتصال را تست کنید</li>
                            <li>دوره‌ها را همگام‌سازی کنید</li>
                            <li>از شورت‌کد <code>[royal_moodle_courses]</code> برای نمایش دوره‌ها استفاده کنید</li>
                            <li>از شورت‌کد <code>[royal_moodle_enroll_button course_id="123"]</code> برای دکمه ثبت‌نام استفاده کنید</li>
                        </ol>
                    </div>
                    
                    <div class="mws-info-box">
                        <h3>شورت‌کدها</h3>
                        <ul>
                            <li><code>[royal_moodle_courses]</code> - نمایش لیست دوره‌ها</li>
                            <li><code>[royal_moodle_courses limit="5"]</code> - نمایش 5 دوره</li>
                            <li><code>[royal_moodle_course moodle_id="123"]</code> - نمایش یک دوره</li>
                            <li><code>[royal_moodle_enroll_button course_id="123"]</code> - دکمه ثبت‌نام</li>
                            <li><code>[royal_moodle_my_courses]</code> - دوره‌های من</li>
                            <li><code>[royal_moodle_user_courses]</code> - <strong>دوره‌های کاربر با گرید و پیشرفت</strong></li>
                            <li><code>[royal_moodle_enrolled_count]</code> - <strong>تعداد دوره‌های ثبت‌نام شده کاربر</strong></li>
                        </ul>
                        <p style="margin-top: 10px; font-size: 12px; color: #666;">
                            <strong>نکته:</strong> شورت‌کدهای قدیمی (<code>moodle_courses</code>, <code>moodle_course</code> و ...) همچنان کار می‌کنند اما توصیه می‌شود از شورت‌کدهای جدید استفاده کنید.
                        </p>
                        <div style="margin-top: 15px; padding: 10px; background: #f0f6fc; border-right: 3px solid #0073aa; border-radius: 4px;">
                            <strong style="color: #0073aa;">شورت‌کد جدید:</strong>
                            <p style="margin: 8px 0 0 0; font-size: 12px; line-height: 1.6;">
                                <code>[royal_moodle_user_courses]</code> دوره‌های ثبت‌نام شده کاربر فعلی را در یک گرید 3 ستونی زیبا نمایش می‌دهد. هر کارت شامل تصویر دوره، عنوان و وضعیت پیشرفت است. با کلیک روی هر کارت، کاربر مستقیماً به صفحه دوره در Moodle هدایت می‌شود.
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
}

